///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file CreationMode.h 
 * \brief Contains the definition of the Core::CreationMode class. 
 */

#ifndef __OVITO_CREATION_MODE_H
#define __OVITO_CREATION_MODE_H

#include <core/Core.h>
#include <core/viewport/input/ViewportInputHandler.h>
#include <core/undo/UndoManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/objects/SceneObject.h>
#include <core/viewport/Viewport.h>

namespace Core {

class PropertiesPanel;		// defined in PropertiesPanel.h

/**
 * \brief Base class for all input modes that create new scene objects.
 * 
 * Plugins that define custom object types should also define and publish a custom input handler 
 * derived from CreationMode to let the user create the object in the viewports.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT CreationMode : public SimpleInputHandler
{
protected:
	
	/// \brief Default constructor.
	CreationMode() : _propertiesPanel(NULL) {}

public:

	/// \brief Sets the properties panel in which the creation mode should display the 
	///        properties of the new objects it creates.
	/// \param panel The properties panel.
	///
	/// This method is called by the system when it activates the creation input mode.
	/// \sa propertiesPanel()
	void setPropertiesPanel(PropertiesPanel* panel) { _propertiesPanel = panel; }

	/// \brief Returns the properties panel in which the creation mode should display the 
	///       properties of the new objects it creates.
	/// \return The properties panel in which the editor of the newly created object should be displayed.
	PropertiesPanel* propertiesPanel() const { return _propertiesPanel; }

private:

	/// The properties panel where the properties of the new object are shown.
	PropertiesPanel* _propertiesPanel;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(CreationMode)
};

/**
 * \brief Basic viewport input mode handler that lets the user create an object.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SimpleCreationMode : public CreationMode
{
public:

	/// Returns the activation behaviour of this input handler.
	virtual InputHandlerType handlerActivationType() { return NORMAL; }

protected:

	/// \brief Constructor.
	/// \param undoDisplayName A string that describes the creation operation.
	/// \param objectBaseName A string that is used to name the newly created object in the scene.
	SimpleCreationMode(const QString& undoDisplayName, const QString& objectBaseName);

	/////////////////////// from SimpleInputHandler ////////////////////////////////
	
	/// \brief This method is called when the user aborts the operation by clicking the right
	///        mouse button or activating another input mode.
	virtual void onAbort();

	/// \brief Is called when the user finishes the operation.
	virtual void onFinish();

	/// \brief Will be called when the user presses the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMousePressed(QMouseEvent* event);

	/// \brief Will be called when the user releases the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMouseReleased(QMouseEvent* event);

	/// \brief Will be called when the user moves the mouse while the operation is active.
	/// \param event The mouse event to be handled.
	virtual void onMouseDrag(QMouseEvent* event);

	/// \brief Will be called when the user moves the mouse while the operation is not active.
	/// \param vp The viewport in which the mouse cursor is located.
	/// \param event The mouse event to be handled.
	virtual void onMouseFreeMove(Viewport& vp, QMouseEvent* event);

protected:
	
	/// \brief Starts a new adjust operation.
	///
	/// This must be called before some aspect or parameter of the new object is interactively
	/// adjusted in the viewport.
	void beginAdjustOperation();

	/// \brief Undoes the current adjust operation and removes it from the undo stack.
	void abortAdjustOperation();

	/// \brief Finishes an adjust operation and puts it on the undo stack.
	void commitAdjustOperation();

	/// \brief This creates the actual scene object.
	/// \return The scene object to be inserted into the scene.
	virtual SceneObject::SmartPtr createObject() = 0;

	/// \brief This creates the scene node for the object.
	/// \param obj The scene object.
	/// \return The newly created scene node to be inserted into the scene.
	virtual ObjectNode::SmartPtr createNode(SceneObject* obj);

	/// \brief Returns the node for the object.
	/// \return The newly created scene node that has been inserted into the scene.
	///
	/// This can be used by derived classes to adjust the position of the node
	/// in the scene.
	ObjectNode* objectNode() const { 
		CHECK_OBJECT_POINTER(objNode.get()); 
		return objNode.get(); 
	}

	/// \brief Returns the object being created.
	/// \return The newly created object.
	///
	/// This can be used by derived classes to adjust the properties of the object
	/// during creation.
	SceneObject* object() const { 
		CHECK_OBJECT_POINTER(obj.get()); 
		return obj.get(); 
	}

	/// \brief Returns the how many times the mouse has been clicked by the 
	///        user during the current operation.
	/// \return The number of mouse clicks since the beginning of the operation.
	int clickCount() const { return _clickCount; }

private:

	/// For undoing the complete create operation.
	CompoundOperation* createOperation;

	/// For undoing the position/resize operations.
	QVector<CompoundOperation*> adjustOperations;

	/// The object node.
	ObjectNode::SmartPtr objNode;

	/// The object.
	SceneObject::SmartPtr obj;

	/// The number of clicks during the operation.
	int _clickCount;

	/// The operation name used for undo.
	QString undoDisplayName;

	/// The base name for new objects created by this mode.
	QString objectBaseName;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(SimpleCreationMode)
};


};

#endif // __OVITO_CREATION_MODE_H
