
import graphe.FGraphCol;
import graphe.FGraphImg;
import graphe.FGraphInfo;
import graphe.GraphCol;
import graphe.GraphImg;
import graphe.GraphInfo;

import java.awt.Color;
import java.awt.image.BufferedImage;

import maths.Angle;
import maths.CoordSyst;
import maths.Difference;
import maths.Disk;
import maths.Exponential;
import maths.FCoordSyst;
import maths.Fonction;
import maths.Homography;
import maths.Intersection;
import maths.Map;
import maths.Polynomial;
import maths.Quotient;
import maths.Rectangle;
import maths.SemiPlane;
import maths.Set;
import maths.Union;

import components.GraphApplet;

/**
 * Ce applet montre une preimage d'ensemble dans une page web.
 *
 * @author Nicolae
 */
public class Preimages extends GraphApplet {

	/**
	 * Le nombre (maximal) d'ensembles.
	 */
	public static final int SET_COUNT = 73;
	
	/**
	 * Le nombre (maximal) d'applications.
	 */
	public static final int MAP_COUNT = 50;

	/**
	 * La precision de la verification des ensembles.
	 */
	public static final int CHECK_STEPS = 25;

	/**
	 * Le rapport minimale entre l'aire de la difference symmetrique et l'aire du plus petit ensemble.
	 */
	public static final double MIN_DIFF = 0.05;

	/**
	 * L'aire minimale (en probabilite) de la difference symmetrique des ensembles distincts.
	 */
	public static final double MIN_AREA = 0.01;
	
	/**
	 * Rien.
	 */
	private static final long serialVersionUID = 1L;

	private static final Map one = new Polynomial(1, 0);
	private static final Map Id = new Polynomial(1, 0, 0, 0);
	private static final Map z2 = new Polynomial(1, 0, 0, 0, 0, 0);
	
	/**
	 * This is the default constructor
	 */
	public Preimages() {
		super();
	}

	/**
	 * Initialisation de <code>graph</code>.
	 */
	protected void graphInit() {
		// le graphe
		FGraphImg grIm = new FGraphImg();
		// la fonction
		grIm.setFonction(getPreimage());
		// les couleurs
		GraphCol col = new FGraphCol(getColor());
		grIm.setColors(col);
		// les coordonees - w et h donees par graph
		grIm.setCoords(getCoords());
		
		getGraph().setGraph(grIm);

		// image de fond
		BufferedImage image = getImage();
		getGraph().setImage(image);
		
		// informations graphiques - syst. de coordonees	
		getGraph().setGrInfo(getInfo(grIm, image != null), null); 
		
		// check sets
		check();
	}

	private BufferedImage getImage() {
		return getImageParam("IMAGE");
	}

	private GraphInfo getInfo(GraphImg gimg, boolean hasImg) {
		Boolean bt = getBoolParam("COORDS");
		if(bt == null && hasImg)
			return null;
		if(bt != null && bt.booleanValue() == false)
			return null;
			
		FGraphInfo gi = new FGraphInfo(Color.black, Color.blue);
		gi.setCoords(gimg.getCoords(), null);

		bt = getBoolParam("UNDER");
		if(bt != null)
			gi.setUnder(bt.booleanValue());

		bt = getBoolParam("CIRCLE");
		if(bt != null)
			gi.setCircle(bt.booleanValue());

		bt = getBoolParam("AXES");
		if(bt != null)
			gi.setAxes(bt.booleanValue());

		bt = getBoolParam("BIG");
		if(bt != null)
			gi.setBig(bt.booleanValue());

		bt = getBoolParam("SMALL");
		if(bt != null)
			gi.setSmall(bt.booleanValue());

		bt = getBoolParam("ARROWS");
		if(bt != null)
			gi.setArrows(bt.booleanValue());

		bt = getBoolParam("TEXT");
		if(bt != null)
			gi.setText(bt.booleanValue());
		
		Double step = getRealParam("STEP");
		if(step != null)
			gi.setBigStep(step.doubleValue());
		
		Integer ratio = getIntParam("RATIO");
		if(ratio != null)
			gi.setBigSmallRatio(ratio.intValue());
		
		return gi;	
	}

	private CoordSyst getCoords() {
		FCoordSyst cs = new FCoordSyst();
		
		Double left = getRealParam("LEFT");
		Double right = getRealParam("RIGHT");
		Double top = getRealParam("TOP");
		Double bottom = getRealParam("BOTTOM");
		
		if(left != null && right != null && top != null && bottom != null)
			cs.setCoord(left.doubleValue(), bottom.doubleValue(), 
					    right.doubleValue(), top.doubleValue());
		else
			cs.setCoord(-3.1, -3.1, 3.1, 3.1);
		
		cs.setRatio(1);
		
		return cs;
	}

	private Fonction getPreimage() {
		int param = getParam() - 1;
		
		Set A = getSet((param % SET_COUNT) + 1);
		if(A == null)
			A = new Difference(new Rectangle(0, 0, 1, 1), new Disk(0.5, 0.5, 0.25));
			
		Map f = getMap((param / SET_COUNT) + 1);
		if(f == null)
			f = new Quotient(one, z2); // 1 / z^2
		
		Set B = f.preImage(A);
		return B.charact();
	}

	/**
	 * Renvoie l'application d'indice <code>map</code>.
	 * 
	 * @param map l'indice.
	 * @return l'application.
	 */
	public Map getMap(int map) {
		switch(map) {
		  case 1 : return Id;  // z
			case 2 : return new Polynomial(-1, 0, 0, 0);  // -z
			case 3 : return new Polynomial(1, 0, 1, 0);  // z + 1
			case 4 : return new Polynomial(1, 0, -1, 0);  // z - 1
			case 5 : return new Polynomial(1, 0, 0, 1);  // z + i
			case 6 : return new Polynomial(1, 0, 0, -1);  // z - i
			case 7 : return new Polynomial(1, 0, -0.5, 1);  // z - 0.5 + i 
			case 8 : return new Polynomial(1, 0, 0.5, -1);  // z + 0.5 - i
			case 9 : return new Polynomial(1, 0, 1, 1);  // z + i +1			
			case 10 : return new Polynomial(0.5, 0, 0, -0.5);  //0.5 z - 0.5 i 
			case 11 : return new Polynomial(0.5,0,-0.5,0);  //0.5 z - 0.5 
			case 12 : return new Polynomial(-2, 0, 1, 0);  // -2 z + 1
			case 13 : return new Polynomial(2, 0, 0, 1);  // 2 z + i
			case 14 : return new Polynomial(-2, 0, 0, 1);  // -2 z + i
			case 15 : return new Polynomial(0, -0.5, 1.5, 0);  //-0.5 i  z + 1.5
			case 16 : return new Polynomial(0, 1, 0, 0) ;   // iz
			case 17 : return new Polynomial(0, 1, -0.5, 0);   //i  z - 0.5
			case 18 : return new Polynomial(0, -1, 0.5, 0);  //-i  z + 0.5
			case 19 : return new Polynomial(0, -2, 0.5, 0);  //-2*i z + 0.5
			case 20 : return new Polynomial(0, 2, 0.5, 0);  //2*i z + 0.5
			
			case 21 : return Id;  // z
			case 22 : return z2;  // z^2
			case 23 : return new Polynomial(2, 0, 0, 0, 0, 0);  //2*z^2 
			case 24 : return new Polynomial(0, 1, 0, 0, 0, 0);  //i*z^2 
			case 25 : return new Polynomial(1, 0, 1, 0, 0, 0);  //z^2+z 
			case 26 : return new Polynomial(1, 0, -1, 0, 0, 0);  //z^2-z 
			case 27 : double re[] = {0, 0, 0, 1};
			         double im[] = {0, 0, 0, 0};
			         return new Polynomial(re, im);  // z^3
			case 28 : double re1[] = {0, 0, 0, 0};
	         double im1[] = {0, 0, 0, 1}; 
	         return new Polynomial(re1, im1);  // iz^3
			case 29 : double re2[] = {0, 0, 0, -1};
	         double im2[] = {0, 0, 0, 0};
	         return new Polynomial(re2, im2);  // -z^3
			case 30 : return new Quotient(one, Id);  // 1 / z
			case 31 : return new Quotient(new Polynomial(0,1), Id);  // i / z
			case 32 : return new Quotient(new Polynomial(0,-1), Id);  // -i / z
			case 33 : return new Quotient(one, z2);  // 1 / z^2
			case 34 : return new Quotient(new Polynomial(0,1), z2);  // i / z^2
			case 35 : return new Quotient(new Polynomial(0,-1), z2);  // -i / z^2
		
			case 100 : return new Exponential();  // exp(z)	
			
			case 41 : return Id ; 
			case 42 : return new Homography(0, 0, -1, 0, 1, 0, 0, 0); // -1/z
			case 43 : return new Homography(0, 0, -1, 0, 1, 0, 1, 0); //-1/(z+1)
			case 44 : return new Homography(1, 0, 1, 0, 1, 0, -1, 0); // (z+1)/(z-1)
			case 45 : return new Homography(1, 0, -1, 0, 1, 0, 0, 1); // (z-1)/(z+1)
			case 46 : return new Homography(1, 0, 0, 1, 1, 0, 0, -1); // (z+i)/(z-i)
			case 47 : return new Homography(1, 0, 0, -1, 1, 0, 0, 1); // (z-i)/(z+i)
			case 48 : return new Homography(0, 1, -1, 0, 1, 0, 1, 0); // (iz-1)/(z+1)
			case 49 : return new Homography(1, 0, -1, 0, 0, 1, 1, 0); // (z-1)/(iz+1)
			case 50 : return new Homography(2, 0, -1, 0, 0, 1, 1, 0); // (2z-1)/(iz+1)
			case 51 : return new Homography(1, 0, -2, 0, 1, 0, 1, 0); // (z-2)/(z-1)
			case 52 : return new Homography(1, 0, -2, 0, 2, 0, 1, 0); // (z+2)/(2z+1)
		}
		
		return null;
	}

	/**
	 * Renvoie l'ensemble d'indice <code>set</code>.
	 * 
	 * @param set l'indice.
	 * @return l'application.
	 */
	public Set getSet(int set) {
		switch(set) {
		  
			case 1 : return new SemiPlane(1, 0, -0.5); 
			case 2 : return new SemiPlane(-1, 0, 0.5);
			case 3 : return new SemiPlane(0, -1, 0.5);
			case 4 : return new SemiPlane(0, 1, 0.5); 
			case 5 : return new SemiPlane(1, -1, 1); 
			case 6 : return new SemiPlane(-1, 1, -1) ;
			case 7 : return new SemiPlane(1, -1, -1); 
			case 8 : return new SemiPlane(-1, 1, 1); 
			case 9 : return new SemiPlane(1, 1, 1); 
			case 10 : return new SemiPlane(-1, -1, -1); 
			case 11 : return new SemiPlane(1, 1, -1); 
			case 12 : return new SemiPlane(-1, -1, 1); 
			
			case 21 : return new Rectangle(-0.25, 0, 0.25, 1.5) ; 
			case 22 : return new Rectangle(0.6, -0.4, 1.4, 0.4); 
			case 23 : return new Intersection(new SemiPlane(1, 0, 0.25),
				         new SemiPlane(-1, 0, 0.75)); 
			case 24 : return new Intersection(new SemiPlane(0, 1, -0.5),
					        new SemiPlane(0,-1, 1)) ; 
			case 25 : return new  Intersection(new SemiPlane(0, 1, -0.25),
					        new SemiPlane(0,-1, 1)); 
			case 26 : return new  Intersection(new SemiPlane(1, 1, 0.5),
					        new SemiPlane(-1,-1, 0.5)); 
			case 27 : return new  Intersection(new SemiPlane(1, 1, 0),
					        new SemiPlane(-1,-1, 0.5)); 
			case 28 : return new Intersection(new SemiPlane(1, -1, 0.5),
					        new SemiPlane(-1,1, 0.5)) ; 
			case 29 : return new Intersection(new SemiPlane(1, -1, 0),
					           new SemiPlane(-1,1, 0.25)) ; 

			case 41 : return new Disk(0.5, 0.5, 0.6); 
			case 42 : return new Disk(1, -1, 0.5); 
			case 43 : return new Difference(new Disk(0, 0, 1),new Disk(0, 0, 0.5)); 
			case 44 : return new Difference(new Disk(0, 0, 1),new SemiPlane(0, 1, -0.5)); 
			case 45 : return new Difference(new Disk(0, 0, 1),new SemiPlane(0, -1, 0.5)); 
			case 46 : return new Difference(new Disk(0, 0, 1),new SemiPlane(-1, 0, 0)); 
			case 47 : return new Difference(new Disk(0, 0, 1),new SemiPlane(1, 0, 0)); 
      case 48 : return new Angle(1, 0.5, 1, 0);                        
      case 49 : return new Intersection(new Angle(1, 1, 1, -1),
				                          new Disk(0, 0, 1)); 
			case 50 : return new Difference(new Disk(0, -1, 1),new Disk(0, -1, 0.5)); 
			case 51 : return new Difference(new Rectangle(0, 0, 2, 2),
					        new Disk(0.5, 0.5, 0.5));  
		   case 52 : return new Union(new Rectangle(0, 0, 2, 2),
					        new Disk(-0.5, -0.5, 0.5));  	
		
			}
		return null;
	}

	/**
	 * Renvoie le parametre numerique de l'applet.
	 *
	 * @return le parametre numerique de l'applet.
	 */
	private int getParam() {
		Integer i = getIntParam("PARAM");
		
		if(i == null)
			return 0;

		return i.intValue();
	}

	/**
	 * Renvoie la couleur donnee en parametre a l'applet.
	 *
	 * @return  la couleur donnee en parametre a l'applet, rouge par default.
	 */
	protected Color getColor() {
		Color col = getColorParam("COLOR");
		if(col == null)
			col = Color.red;
		
		return col;
	}
	
	private void check() {
		Boolean chk = getBoolParam("CHECK");
		if(chk == null || ! chk.booleanValue())
			return;

		int chks = 0;
		
		CoordSyst coo = getCoords();
		Integer steps = getIntParam("CHECK_STEPS");
		int step = CHECK_STEPS;
		if(steps != null)
			step = steps.intValue();
			
		coo.setPix(step, step);
		
		double mina = MIN_AREA;
		double mind = MIN_DIFF;
		Double minp = getRealParam("CHECK_MIN_AREA");
		if(minp != null)
			mina = minp.doubleValue();
		minp = getRealParam("CHECK_MIN_DIFF");
		if(minp != null)
			mind = minp.doubleValue();

		double eps = 1;
		eps /= CHECK_STEPS;
		eps /= CHECK_STEPS;
		
		for (int j = 1; j <= MAP_COUNT; j++) {
			Map f1 = getMap(j);
			if(f1 == null)
				continue;

			for (int i = 1; i <= SET_COUNT; i++) {
				Set s1 = getSet(i);
				if(s1 == null)
					continue;

				Set p1 = f1.preImage(s1);
				double a1 = getArea(p1, coo);
				chks ++;
				if(a1 < mina)
					System.out.println("L'ensemble [" + j + ", " +  i + "] est (presque) vide !");
				boolean first = true;

				for (int l = j; l <= MAP_COUNT; l++) {
					Map f2 = getMap(l);
					if(f2 == null)
						continue;

					for (int k = (j < l ? 1 : i + 1); k <= SET_COUNT; k++) 
					  if ( i == k || j == l ){
						Set s2 = getSet(k);
						if(s2 == null)
							continue;

						Set p2 = f2.preImage(s2);
						double area = getArea(p1.symmDiff(p2), coo);
						chks ++;
						
						if(a1 < eps)
							continue;
						
						if(area / a1 < mind) {
							if(first) {
								System.out.print("ensembles presque identiques :  [" + j + ", " +  i + 
										"], [" +
										l + ", " + k + "]");
								first = false;
							} else 
								System.out.print(" , [" + l + ", " + k + "]");
						}
					}
				}
				
				if(! first)
					System.out.println();
			}
		}

		System.out.println();
		System.out.println(chks + " ensembles (ou paires) verifis (" + 
				step * step + " tests chacun) !");
		System.out.println("Details : MIN_AREA = " + mina + ", MIN_DIFF = " + mind);
	}

	private double getArea(Set set, CoordSyst coo) {
		int area = 0;
		for (int i = 0; i < coo.getWidth(); i++) {
			for (int j = 0; j < coo.getHeigth(); j++) {
				if(set.contains(coo.getRe(i), coo.getIm(j)))
					area ++;
			}
		}
		
		double a = area;
		a /= coo.getWidth();
		a /= coo.getHeigth();
		
		return a;
	}
}  //  @jve:decl-index=0:visual-constraint="10,10"
