;;; MACROS
;;;

; To see what these macros expand out to, use:
;  (macroexpand '(<macro> <macro-args>) '() state)
; or 
;  (expand (<macro> <args>))

;; (expand term)
;; expands macro to one level
(defmacro expand (macro)
  `(macroexpand1 (quote ,macro) '() state))

;;(begin list[code]) -> list[len(list)]
;;   returns results of last expression
(defmacro begin (exp &rest rst)
  (cond ((endp rst) exp)
        ((endp (cdr rst)) `(prog2$ ,exp ,(car rst)))
        (t `(prog2$
             ,exp
             (begin ,(car rst) ,@(cdr rst))))))

;;(lookup string [string]) -> value
;; macro to lookup variable in current env
(defmacro lookup (str1 &optional str2)
  `(if (not (null ,str2)) 
       (get-hash-var ,str2 ,str1 env)
     (get-var ,str1 env)))

;;(lookup string [string]) -> value
;; macro to lookup variable in current env
(defmacro lookupx (str &optional str2)
  (let ((s1 (string str))
        (s2 (string str2)))
    `(if (not (null ,s2)) (get-hash-var ,s2 ,s1 env)
       (get-var ,s1 env))))


;;(let-seq var list[code]) -> list[len(list)]
;; Sequentially binds var to results of list[i] code
;;  returns results of last expression
(defmacro let-seq (var &rest rst)
  (cond ((endp rst) (car rst))
        ((endp (cdr rst)) (car rst))
        (t `(let ((,var (,@(car rst))))
              (let-seq ,var ,@(cdr rst))))))

; mv-let-seq var list[code] -> list[len(list)]
; sequentially binds var to results of list[i] code
;  returns results of last expression
(defmacro mv-let-seq (vars &rest rst)
  (cond ((endp rst) (car rst))
        ((endp (cdr rst)) (car rst))
        (t `(mv-let ,vars (,@(car rst))
              (mv-let-seq ,vars ,@(cdr rst))))))

; mv-let-seq var list[code] -> list[len(list)]
; sequentially binds var to results of list[i] code
;  returns results of last expression
(defmacro mv-let-seq-use (vars &rest rst)
  (cond ((endp rst)       `())
        ((endp (cdr rst)) `(begin ,@vars (,@(car rst))))
        (t `(mv-let ,vars  (begin ,@vars (,@(car rst)))
              (mv-let-seq-use ,vars ,@(cdr rst))))))



;;(defalias f1 f2) -> (defmacro f1 ...)
;;defines alias for function
;;Macro-defining macro from "On Lisp" by Paul Graham
(defmacro defalias (f1 f2)
  `(defmacro ,f1 (&rest args)
     `(,',f2 ,@args)))

;;(fif feature if_stmt else_stmt) -> statement
;;if feature is enabled, use if_stmt, otherwise else_stmt
(defmacro fif (feature if_stmt &optional else_stmt)
  (cond ((member-equal feature *features-present*)
           `,if_stmt)
        (t `,else_stmt)))

;;(fif-all features if_stmt else_stmt) -> statement
;;if features are enabled, use if_stmt, otherwise else_stmt
(defmacro fif-all (features if_stmt &optional else_stmt)
  (cond ((null features) `,if_stmt)
	((member-equal (car features) *features-present*)
	 (let ((f2 (cdr features)))
	   `(fif-all ,f2 ,if_stmt ,else_stmt)))
        (t `,else_stmt)))

;;(fif-any features if_stmt else_stmt) -> statement
;;if any feature is enabled, use if_stmt, otherwise else_stmt
(defmacro fif-any (features if_stmt &optional else_stmt)
  (cond ((null features) (begin (cw "(t: ~x0)~%" if_stmt) `,else_stmt))
	((member-equal (car features) *features-present*)
	 `,if_stmt)
        (t 
	 (let ((f2 (cdr features)))
	       `(fif-any ,f2 ,if_stmt ,else_stmt)))))

;;; ENV FUNCTIONS
(defun envp (env)
 (declare (xargs :guard t))
  (symbol-alistp env))
   
;;(set-var symbol value env) -> env
;; Returns a new list with 'var changed to 'val
(defun set-var (var val env)
;  (declare (xargs :guard (and (symbolp var)
;			      (symbol-alistp env)
;			  )))
  (acons var val env))

;;(get-var symbol env) -> value
;; Return val of 'var in list
(defun get-var (var env)
  (declare (xargs :guard (and (symbolp var)
			      (symbol-alistp env))))
  (cdr (assoc var env)))

;;(has-var symbol env)-> boolean
;; checks whether symbol is defined in alop
;(defalias has-var assoc)
(defun has-var (var env)
  (declare (xargs :guard (and (symbolp var)
			      (symbol-alistp env))))
  (assoc var env))

(defun symbol2-alist2p (var1 var2 alist)
  (declare (xargs :guard t)) ;(and (symbolp var1)
			     ; (symbolp var2)
			     ; (symbol-alistp alist))))
  (and (symbol-alistp alist)
       (symbolp var1)
       (let ((alist2 (get-var var1 alist)))
	 (and 
	  (symbolp var2)
	  (symbol-alistp alist2)))))

;;(set-hash-var symbol value symbol env) -> env
;; sets var to val in hash contained in env
(defun set-hash-var (var val hash env)
;  (declare (xargs :guard (and (symbolp var)
;			      (symbolp hash)
;			      (symbol-alistp env)
;			      (symbol2-alist2p hash var env)
;			      )))
  (set-var hash                              ; Set hash var
           (set-var var val                  ;  after setting var->val
                    (get-var hash env)) env);  after getting hash
  )

;;(get-hash-var symbol symbol env) -> value
;; get value of var in hash contained in env
(defun get-hash-var (var hash env)
  (declare (xargs :guard (and (symbolp var)
			      (symbol-alistp env)
			      (symbol2-alist2p hash var env))))
  (get-var var                 ; Set hash var 
           (get-var hash env))) ;  after getting hash

(defun host-envp (env)
 (declare (xargs :guard t))
 (and (envp env)
      (atom-listp (lookup 'users))))

(defun user-envp (env)
 (declare (xargs :guard t))
 (and (envp env)
      (listp (lookup 'already-answered))))


(defun env-subset (?sub ?super)
  (declare (xargs :guard (and
			  (symbol-alistp ?super)
			  (symbol-alistp ?sub))))
   (subsetp-equal (domain ?sub) (domain ?super)))

;;(comment string env) -> env
;; prints out a comments, returns env
(defun comment (str env)
  (declare (xargs :guard (and (stringp str)
			      (symbol-alistp env))))
  (or
   (cw str)   ; returns '()
   (cw "~%")  ; newline
   env))

;;(actionp message) -> boolean
; predicate for checking whether variable is an action
(defun actionp (act)
  (declare (xargs :guard t))
  (and (symbol-alistp   act)
       (get-var 'command act)
       (has-var 'args    act)
       (has-var 'env     act)))


;;; HELPER FUNCTIONS
;;; 

;;(stringify-helper <anything> boolean -> string
;; helper function to stringify
(defun stringify-helper (list in-list?)
  (declare (xargs :guard t))
  (cond ((null list) ")")
	((stringp list)    (string list))
	((characterp list) (string list))
	((symbolp list)    (string list))
	((consp   list)
	 (string-append 
	  (if in-list? " " "(")
	  (string-append 
	   (stringify-helper (car list) nil)
	   (string-append " "
			  (stringify-helper (cdr list) t)))))
	(t "<error>")))


;;(stringify ?) -> string
;; stringifies list or variable
(defun stringify (list)
  (declare (xargs :guard t))
  (stringify-helper list nil))

;;(los-adjoin item list) -> list
;; Add item to list if item is not already part of list
(defun los-adjoin (item list)
  (declare (xargs :guard (and (atom item)
			      (atom-listp list))))
  (if (member-equal item list)
      list
    (cons item list)))

(defstructure email
  (user (:assert (and (symbolp user)
		      (not (null user)))))
  (host (:assert (and (symbolp host)
		      (not (null host)))))
  (:options :guards))

(defun email-listp (lst)
  (declare (xargs :guard t))
  (if (ATOM LST) 
      (EQ LST NIL)
    (AND (email-P (CAR LST))
	 (email-LISTP (CDR LST)))))

(defun recips-p (r)
  (declare (xargs :guard t))
  (and (listp r)
       (email-p (car r))
       (email-listp (cdr r))))

(defstructure message
  (sender  (:assert (email-p sender)))
  (recips  (:assert (recips-p recips)))
  (headers (:assert (symbol-alistp headers)))
  (body    (:assert (listp body)))
  (:options :guards)
)

(defun recipient (msg)
  (declare (xargs :guard (and (message-p msg)
			      (not (null (message-recips msg))))))
  (car (message-recips msg)))

(defmacro mk-email (user host)
  `(make-email :user ,user :host ,host))

(defmacro mk-message (s r h b)
  (if (listp b)
      `(make-message :sender ,s
		     :recips ,r
		     :headers (set-var 'from ,s
				       (set-var   'to   ,r ,h))
		     :body   ,b)
          `(make-message :sender ,s
		     :recips ,r
		     :headers (set-var 'from ,s
				       (set-var   'to   ,r ,h))
		     :body   '(,b))))


;;(wrap-msg-body message symbol -> message)
;; wraps body of message in wrapper, for example, encryption
(defun wrap-msg-body (msg wrapper)
;  (declare (xargs :guard (and (messagep msg)
;			      (symbolp       wrapper))))
  (update-message msg
		  :body 
		  (list wrapper (message-body msg))))

;;(get-msg-wrapper message) -> symbol
;; gets the wrapper around the message
(defun get-msg-wrapper (msg)
  (begin (cw "~x0 ~%" (stringify msg))
	 (let ((wrapper (message-body msg)))
	   (and (consp wrapper)
		(car wrapper)))))


;;(get-msg-header string message -> header
;; get the header which contains 
(defun get-msg-header (header msg)
  (get-var header (message-headers msg)))

;;(set-msg-header string string message -> message
;; add (replace) the header starting with <header> with <value>
(defun set-msg-header (header value msg)
  (update-message msg 
		  :headers
		  (set-var header value (message-headers msg))))
	       
;;(subject message) -> string
;; returns subject of message
(defun subject (msg)
  (get-msg-header 'subject msg))


(defun ret3 (status arg env) 
  (declare (xargs :guard (and (symbolp status)
			      (message-p arg)
			      (symbol-alistp env))))
  (mv status arg env))

;; (return3 symbol msg env) -> (mv symbol msg env)
;;  syntactical sugar to return from feature
(defmacro act (status &optional arg env)
  (if (equal 'comment status)
      `(begin 
	(cw "Comment: ~x0~%" ,arg) ; output comment
	(ret3 ',status msg env))
    (if env
	`(ret3 ',status ,arg ,env)
	`(ret3 ',status ,arg env))))

(defun action-commandp (action)
  (and (listp action)
       (listp (cdr action))
       (listp (cddr action))
       (listp (cdddr action))
  (let ((who  (car action))
	(what (cadr action))
	(arg1 (caddr action))
	(arg2 (cadddr action)))
    (and (or (has-var who *hosts*)
	     (has-var who *users*))
	 (equal 'command what)
	 (and (symbolp arg1)
	      (memberp arg1 *commands*))
	 (listp arg2)))))

(defun msg-type-p (msg)
  (declare (xargs :guard t))
  (and (listp msg)
       (listp (cdr msg))
       (listp (cddr msg))
       (listp (cdddr msg))
       (listp (cddddr msg))
       (let ((head (car msg))
	     (from (cadr msg))
	     (to   (cadddr msg))
	     (body (cadddr (cdr msg))))
	 (and 
	  (equal 'smessage head)
	  (listp from)
	  (listp to)
	  (or (stringp body)
	      (atom    body)
	      (listp   body))))))

(defun action-sendp (action)
  (and (listp action)
       (listp (cdr action))
       (listp (cddr action))
       (let ((who  (car action))
	     (what (cadr action))
	     (msg  (caddr action)))
	 (begin (cw "~x0 - ~x1 - ~x2~%" who what msg)
	 (and (or (has-var who *hosts*)
		  (has-var who *users*))
	      (equal 'send what)
	      (msg-type-p msg))))))

(defun actions-helper (actions status)
  (if (null actions) 
      (equal 'end status)
    (if (listp actions)
	 (let ((action (car actions))
	       (rest   (cdr actions)))
	   (if (equal 'init status)
	       (if (equal 'init (car action))
		   (actions-helper rest 'command)
		 nil)
	     (if (equal 'command status)
		 (if (action-commandp action)
		     (actions-helper rest 'command)
		   (if (action-sendp action)
		       (actions-helper rest 'end)
		     nil))
	       nil)))
      nil)))

(defun actions-listp (actions)
  (begin
  (actions-helper actions 'init)
  )
)

(defun get-vars (vals alist)
  (cond ((null vals) nil)
	(t (and (listp vals) 
		(let ((val (car vals)))
		  (acons
		   val
		   (get-var val alist)
		   (get-vars (cdr vals) alist)))))))

(defun equal-or-one-off (l1 l2)
  (or (equal l1 l2)
      (equal (cdr l1) l2)))

(defun wrap-list (lst wrapper)
  (cons wrapper lst))

(defun unwrap-list (lst wrapper)
  (if (and (consp lst)
             (equal wrapper
                    (car lst)))
      (cdr lst)
    lst))

(defmacro mv-status  (lst) `(mv-nth 0 ,lst))
(defmacro mv-msg     (lst) `(mv-nth 1 ,lst))
(defmacro mv-env     (lst) `(mv-nth 2 ,lst))

(defmacro mv-actions (lst) `(mv-nth 0 ,lst))
(defmacro mv-users   (lst) `(mv-nth 1 ,lst))
(defmacro mv-hosts   (lst) `(mv-nth 2 ,lst))

(defmacro action-msg (act) `(action-arg1 ,act))
