/* $Id: kmo_wave_cal-test.c,v 1.22 2013-10-08 11:18:57 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:57 $
 * $Revision: 1.22 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>
#include <math.h>

#include <cpl.h>

#include "kmclipm_functions.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_wave_cal/";

const char      *valid_files[]      = {"arc_on_123.fits",
                                       "arc_off_123.fits",
                                       "Ar_prediction_K.txt",
                                       "kmos_ar_k_full_sim.fits",
                                       "kmos_wave_ref_table_test2.fits",
                                       "kmos_wave_band.fits"};

/**
    @defgroup kmo_wave_cal   kmo_wave_cal unit tests

    @{
*/

/**
 * @brief test with data and mask
 */
static void test_wave_cal(const char *path_op_on1,
                          const char *path_op_off1,
                          int ret_val)
{
    // create sof-file
    const char *sof_path = "test_wave_cal.sof";
    FILE *fh = fopen(sof_path, "w");
    char *my_path = cpl_sprintf("%s/ref_data", getenv("srcdir"));
    fprintf (fh,
             "%s                            ARC_ON\n"
             "%s                            ARC_OFF\n"
             "badpixel_flat_KKK.fits        BADPIXEL_FLAT\n"
             "xcal_KKK.fits                 XCAL\n"
             "ycal_KKK.fits                 YCAL\n"
             "flat_edge_KKK.fits            FLAT_EDGE\n"
             "master_flat_KKK.fits          MASTER_FLAT\n"
             "%s/%s                   ARC_LIST\n"
             "%s/%s                   REF_LINES\n"
             "%s/%s                   WAVE_BAND\n",
             path_op_on1, path_op_off1,
             my_path, valid_files[3], my_path, valid_files[4], my_path, valid_files[5]);
    fclose(fh);
    cpl_free(my_path);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_wave_cal --b_start=1.948 "
                                  "--b_end=2.351 --dev_disp=0.0002 "
                                  "--dev_flip=TRUE %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_wave_cal.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
  @brief    Test of kmo_wave_cal recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_wave_cal-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_wave_cal-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            file_path[256],
            tmp[256],
            op1_file_on[256],
            op1_file_off[256];

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    // get rid of warning...
    if ((argc == 5) && (strcmp(argv[1], "xxx") != 0)) {
    }

    FILE *fd = fopen("log_kmo_wave_cal.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    //
    //  PIPELINE TEST (must be called after kmo_flat-test, since the
    //  following recipes rely on this output)*/
    //
    cpl_msg_info(cpl_func, "Testing kmo_wave_cal with real pipeline data.");
    sprintf(test_path, "%s%s", test_global_path_test_data, "pipeline/");

    kmo_test_cat_strings(op1_file_on, test_path, valid_files[0]);
    kmo_test_cat_strings(op1_file_off, test_path, valid_files[1]);
    test_wave_cal(op1_file_on, op1_file_off, 0);

    // check LCAL
    strcpy(tmp, LCAL);
    sprintf(file_path, "%s_KKK",kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 6.45113, tol*10);
    strcat(file_path, ".fits");

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(cpl_propertylist_get_int(h, QC_ARC_SAT), 372);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_EFF),
                 17.3546, tol*10);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_POS_MEAN),
                 0.222684, tol*20);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_POS_MAXDIFF),
                 0.486981, tol*22);
    cpl_test_abs(cpl_propertylist_get_int(h, QC_ARC_AR_POS_MAXDIFF_ID),
                 7, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_POS_STDEV),
                 1.76018, tol*3);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_POS_95ILE),
                 3.74018, tol*5);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_FWHM_MEAN),
                 45.2934, tol*7);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_FWHM_STDEV),
                 3.95828, tol*4);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ARC_AR_FWHM_95ILE),
                 51.2953, tol*40);
    cpl_propertylist_delete(h); h = NULL;

    // check det_img_wave
    strcpy(file_path, "det_img_wave_KKK");
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 327.862, tol*15);

    strcat(file_path, ".fits");
    h = kmclipm_propertylist_load(file_path, 1);
// ADD HERE THE OTHER QC PARAMETERS
    cpl_propertylist_delete(h); h = NULL;

    return cpl_test_end(0);
}

/** @} */
