# Application Insights DotNet SDK Concepts
This lists the high level concepts of the AI DotNet SDK and links to detailed guides to help you get started.

To use the Application Insights SDK you must configure an Instrumentation Key which can be [obtained from an Application Insights resource](https://docs.microsoft.com/azure/azure-monitor/app/create-new-resource).
Or you can use a [Connection string](https://docs.microsoft.com/azure/azure-monitor/app/sdk-connection-string?tabs=net) to simplify the configuration of our endpoints.
Both an Instrumentation Key and Connection String are provided for you on the Overivew Dashboard of your Application Insights resource.

## TelemetryClient
The `TelemetryClient` object is the primary root object for the library. 
Almost all functionality around telemetry sending is located on this object. 

### Initialization
You must initialize an instance of this object and populate it with your Instrumentation Key to identify your data.

```C#
using Microsoft.ApplicationInsights;
using Microsoft.ApplicationInsights.Extensibility;

var configuration = new TelemetryConfiguration
{
    ConnectionString = "InstrumentationKey=00000000-0000-0000-0000-000000000000",
};
var tc = new TelemetryClient(configuration);
```

### Using the TelemetryClient to send telemetry
You can populate common context on the `TelemetryClient.context` property which will be automatically attached to each telemetry item sent. 
You can also attach additional property data to each telemetry item sent. 
The `TelemetryClient` also exposes several `Track` methods that can be used to send all telemetry types understood by the Application Insights service. Some example use cases are shown below.

Please review the full [API summary for custom events and metrics](https://docs.microsoft.com/azure/azure-monitor/app/api-custom-events-metrics) for more examples.

```C#
tc.Context.Device.OperatingSystem = Environment.OSVersion.ToString();

tc.TrackTrace(message: "Custom message.");

tc.TrackEvent(
    eventName: "PurchaseOrderSubmitted", 
    properties: new Dictionary<string, string>() { { "CouponCode", "JULY2015" } }, 
    metrics: new Dictionary<string, double>() { { "OrderTotal", 68.99 }, { "ItemsOrdered", 5 } }
    );
	
try
{
    // do something
}
catch(Exception ex)
{
    tc.TrackException(ex);
}
``` 

### Flushing
When your application is shutting down, we recommend Flushing the `TelemetryClient` to clear the buffer and prevent telemetry from being lost.

Please review our examples here: [Flushing data](https://learn.microsoft.com/azure/azure-monitor/app/api-custom-events-metrics#flushing-data).

## Telemetry Channels
Telemetry channels are responsible for sending the telemetry data to the designated place. Optional features can be provided by the telemetry channels, for example, buffering the data and sending in them in batches, persisting the data to a local storage in case of transmission failure (e.g. network outage), traffic shaping and retry mechanisms

The .NET and .NET Core versions of the SDKs provide two built-in telemetry channels:
- [InMemoryChannel](https://github.com/microsoft/ApplicationInsights-dotnet/blob/develop/BASE/src/Microsoft.ApplicationInsights/Channel/InMemoryChannel.cs): A lightweight channel that buffers items in memory until they're sent. Items are buffered in memory and flushed once every 30 seconds, or whenever 500 items are buffered.
- [ServerTelemetryChannel](https://github.com/microsoft/ApplicationInsights-dotnet/blob/develop/BASE/src/ServerTelemetryChannel/ServerTelemetryChannel.cs): A more advanced channel that has retry policies and the capability to store data on a local disk.

Please review our full guide on [Telemetry Channels in Application Insights](https://docs.microsoft.com/azure/azure-monitor/app/telemetry-channels).

## TelemetryProcessors and TelemetryInitializers
You can write and configure plug-ins for the Application Insights SDK to customize how telemetry can be enriched and processed before it's sent to the Application Insights service.

Please review our full guide on [Filtering and preprocessing telemetry](https://docs.microsoft.com/azure/azure-monitor/app/api-filtering-sampling).

## Telemetry correlation
Application Insights supports distributed telemetry correlation, which you use to detect which component is responsible for failures or performance degradation.

Please review our full guide on [Telemetry correlation in Application Insights](https://docs.microsoft.com/azure/azure-monitor/app/correlation).

## Custom metric collection
The Azure Monitor Application Insights .NET and .NET Core SDKs have two different methods of collecting custom metrics, `TrackMetric()`, and `GetMetric()`. The key difference between these two methods is local aggregation.

Please review our full guide on [Custom metric collection in .NET and .NET Core](https://docs.microsoft.com/azure/azure-monitor/app/get-metric).

## Sampling
Sampling is the recommended way to reduce telemetry traffic, data costs, and storage costs, while preserving a statistically correct analysis of application data.

Please review our full guide on [Sampling in Application Insights](https://docs.microsoft.com/azure/azure-monitor/app/sampling).
