/* $Id: ErrorRegistry.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __ERROR_REGISTRY_HPP_INCLUDED
#define __ERROR_REGISTRY_HPP_INCLUDED

#include <list>
#include "frontend/reporting/CompileError.hpp"

namespace ast {

//! register errors or warnings
/** register errors or warnings during a SemanticCheck. */
class ErrorRegistry {
public:
	/** register an error
	 *  @param error error that should get registered.
	 */
	static void addError(CompileError *error);
	/** register a warning.
	 *  @param warning warning that should get registered.
	 */
	static void addWarning(CompileError *warning);

	/** register a potential error
	 *  @param error potential error 
	 */
	static void addPotentialError(CompileError* error);

	/** clear all potential errors.
	 */
	static void rejectPotentials(void);

	/** move all potential to the error queue, which means that these
	 *  have indeed been errors.
	 */
	static void acceptPotentials(void);

	/** have any errors been reported yet?
	 *  @return true if errors have been reported.
	 */
	static bool hasErrors(void);

	/** have any warnings been reported yet?
	 *  @return true, if there were warnings.
	 */
	static bool hasWarnings(void);

	/** put all warnings to stream.
	 *  @param stream stream to which warnings should get put to.
	 */
	static void putWarnings(std::ostream &stream);
	
	/** put all errors to stream.
	 *  @param stream stream to which errors should get put to.
	 */
	static void putErrors(std::ostream &stream);

	/** release all reported errors and warnings and free memory. */
	static void flushAll(void);

	/** treat Warnings as errors (-Werror). Must only be set if no
	 *  warnings have been registered yet.
	 *  @param val true, to treat Warnings as errors, false otherwise.
	 */
	static void setWerror(bool val);

private:
	/** compile errors */
	static std::list<CompileError*> errors;
	/** compile warnings */
	static std::list<CompileError*> warnings;
	/** potential errors */
	static std::list<CompileError*> potentialErrors;
	/** -Werror set? */
	static bool werror;
};

}; /* namespace ast */

#endif /* __ERROR_REGISTRY_HPP_INCLUDED */
