/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __PROCESS_HPP_INCLUDED
#define __PROCESS_HPP_INCLUDED

#include <list>
#include "frontend/ast/ConcurrentStat.hpp"
#include "frontend/ast/SeqStat.hpp"
#include "frontend/ast/Name.hpp"
#include "frontend/misc/Driver.hpp"

namespace ast {

//! a VHDL process.
/** This class defines one VHDL process.
 */
class Process : public ConcurrentStat {
public:
	//! c'tor
	/** @param sensitivities list of sensitivities.
          * @param decls list of declarations.
          * @param seqStatL list of sequential Statements.
          * @param loc location of process declaration.
          */
	Process(
		std::list<Name*> *sensitivities,
		std::list<SymbolDeclaration*> *decls,
		std::list<SeqStat*> *seqStatL,
		Location loc
		) : 	ConcurrentStat(loc),
			sensitivityList(sensitivities),
			declarations(decls),
			seqStats(seqStatL) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor &visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << "PROCESS";
		if (this->name != NULL) {
			stream << " \"" << *this->name << '"';
		}
	}

	/** sensitivity list */
	std::list<Name*> *sensitivityList;
	/** local declarations */
	std::list<SymbolDeclaration*> *declarations;
	/** sequential statements */
	std::list<SeqStat*> *seqStats;
	/** list of drivers */
	std::list<Driver*> drivers;

protected:
	/** Destructor */
	virtual ~Process() {
		util::MiscUtil::lterminate(sensitivityList);
		util::MiscUtil::lterminate(declarations);
		util::MiscUtil::lterminate(seqStats);
	}
};

}; /* namespace ast */

#endif /* __PROCESS_HPP_INCLUDED */
