package junit.test.svg;

import junit.framework.TestCase;
import latexDraw.parsers.svg.SVGAttr;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;

import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/** 
 * This class contains tests for the SVGAttr class.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 03/29/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class TestSVGAttr extends TestCase
{
	protected SVGElement node;
	
	
	@Override
	@Before
	public void setUp()
	{
		SVGDocument doc = new SVGDocument();
        node = (SVGElement)doc.createElement("tag1");
	}
	
	
	
	@Test
	public void testConstructor()
	{
		try 
		{ 
			new SVGAttr(null, null, null);
			fail();
		}
		catch(IllegalArgumentException e) { /* ok */ }
		
		try 
		{ 
			new SVGAttr("", "", null);
			fail();
		}
		catch(IllegalArgumentException e) { /* ok */ }
		
		try 
		{ 
			new SVGAttr(null, "", node);
			fail();
		}
		catch(IllegalArgumentException e) { /* ok */ }
		
		try 
		{ 
			new SVGAttr("", null, node);
			fail();
		}
		catch(IllegalArgumentException e) { /* ok */ }
		
		new SVGAttr("", "", node);
	}
	
	
	@Test
	public void testGetName()
	{
		SVGAttr attr = new SVGAttr("attrName", "", node);
		assertEquals("attrName", attr.getName());
	}
	
	
	@Test
	public void testGetElementOwner()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertEquals(node, attr.getOwnerElement());
	}
	
	
	
	@Test
	public void testGetValue()
	{
		SVGAttr attr = new SVGAttr("", "attrValue", node);
		assertEquals("attrValue", attr.getValue());
	}
	
	
	@Test
	public void testIsId()
	{
		SVGAttr attr = new SVGAttr("id", "", node);
		
		assertTrue(attr.isId());
		attr = new SVGAttr("", "", node);
		assertFalse(attr.isId());
	}
	
	
	@Test
	public void testSetValue()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		
		try
		{
			attr.setValue(null);
			fail();
		}
		catch(DOMException e) { /* Ok */ }
		
		attr.setValue("val");
		assertEquals("val", attr.getValue());
	}
	
	
	@Test
	public void testCloneNode()
	{
		SVGAttr attr = new SVGAttr("n", "v", node);
		SVGAttr attr2 = (SVGAttr)attr.cloneNode(false);
		
		assertNotNull(attr2);
		assertEquals(attr.getName(), attr2.getName());
		assertEquals(attr.getValue(), attr2.getValue());
		assertEquals(attr.getOwnerElement(), attr2.getOwnerElement());
	}
	
	
	@Test
	public void testGetNodeName()
	{
		SVGAttr attr = new SVGAttr("attrNodeName", "", node);
		assertEquals("attrNodeName", attr.getNodeName());
	}
	
	
	@Test
	public void testGetNodeType()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertEquals(Node.ATTRIBUTE_NODE, attr.getNodeType());
	}
	
	
	@Test
	public void testGetNodeValue()
	{
		SVGAttr attr = new SVGAttr("", "attrValue", node);
		assertEquals("attrValue", attr.getNodeValue());
	}
	
	
	@Test
	public void testGetParentNode()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertEquals(node, attr.getParentNode());
	}
	
	
	@Test
	public void testHasAttribute()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertFalse(attr.hasAttributes());
	}
	
	
	@Test
	public void testChildNodes()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertFalse(attr.hasChildNodes());
	}
	
	
	@Test
	public void testIsEqualNode()
	{
		SVGAttr attr = new SVGAttr("n", "v", node);
		SVGAttr attr2 = (SVGAttr)attr.cloneNode(false);
		
		assertTrue(attr.isEqualNode(attr2));
		attr2 = new SVGAttr("n", "", node);
		assertFalse(attr.isEqualNode(attr2));
		attr2 = new SVGAttr("", "v", node);
		assertFalse(attr.isEqualNode(attr2));
		assertFalse(attr.isEqualNode(null));
	}
	
	
	@Test
	public void testIsSameNode()
	{
		SVGAttr attr = new SVGAttr("n", "v", node);
		SVGAttr attr2 = (SVGAttr)attr.cloneNode(false);
		
		assertTrue(attr.isSameNode(attr));
		assertFalse(attr.isSameNode(null));
		assertFalse(attr.isSameNode(attr2));
	}
	
	
	@Test
	public void testSetNodeValue()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		
		try
		{
			attr.setNodeValue(null);
			fail();
		}
		catch(DOMException e) { /* Ok */ }
		
		attr.setNodeValue("val");
		assertEquals("val", attr.getValue());
		assertEquals("val", attr.getNodeValue());
	}
	
	
	@Test
	public void testGetPrefix()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		
		assertNull(attr.getPrefix());
		attr = new SVGAttr("pref:", "", node);
		assertEquals("pref", attr.getPrefix());
		attr = new SVGAttr(":", "", node);
		assertEquals("", attr.getPrefix());
	}
	
	
	@Test
	public void testGetNamespaceURI()
	{
		SVGAttr attr   = new SVGAttr("pref:n", "", node);
		SVGElement elt = (SVGElement)node.getOwnerDocument().createElement("tag2");
		
		assertNull(attr.getNamespaceURI());
		elt.setAttribute("xmlns:pref", "namespace");
		elt.appendChild(node);
		assertEquals(attr.getNamespaceURI(), "namespace");
	}
	
	
	@Test
	public void testLookupNamespaceURI()
	{
		SVGAttr attr   = new SVGAttr("pref:n", "", node);
		SVGElement elt = (SVGElement)node.getOwnerDocument().createElement("tag2");
		
		assertNull(attr.getNamespaceURI());
		elt.setAttribute("xmlns:pref", "namespace");
		elt.appendChild(node);
		assertEquals(attr.getNamespaceURI(), "namespace");
	}
	
	
	@Test
	public void testGetLocalName()
	{
		SVGAttr attr = new SVGAttr("attrNodeName", "", node);
		assertEquals("attrNodeName", attr.getLocalName());
	}
	
	
	@Test
	public void testUselessMethods()
	{
		SVGAttr attr = new SVGAttr("", "", node);
		assertNull(attr.removeChild(null));
		assertNull(attr.insertBefore(null, null));
		assertNull(attr.getAttributes());
		assertNull(attr.getFirstChild());
		assertNull(attr.getLastChild());
		assertNull(attr.appendChild(null));
	}
}
