/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.managed.factory.support.beans;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlValue;

import org.jboss.util.JBossStringBuilder;
import org.jboss.managed.api.annotation.ManagementProperty;

/**
 * Metadata for a property.
 * 
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 76911 $
 */
@XmlType(name="propertyType", propOrder={"annotations", "value"})
public class APropertyMetaData extends AFeatureMetaData
   implements IPropertyMetaData, Serializable
{
   private static final long serialVersionUID = 3L;

   /** The property name */
   protected String name;

   /** The preinstantiate */
   protected boolean preInstantiate = true;

   /** The property value */
   protected IValueMetaData value;

   /** The property type */
   protected String type;
   
   /** The property replace */
   private Boolean replace;
   
   /** Whether to trim */
   private Boolean trim;
   
   /**
    * Create a new property meta data
    */
   public APropertyMetaData()
   {
   }

   /**
    * Create a new property meta data
    * 
    * @param name the name
    * @param value the value
    */
   public APropertyMetaData(String name, Object value)
   {
      this.name = name;
      this.value = new AbstractValueMetaData(value);
   }

   /**
    * Create a new property meta data
    * 
    * @param name the name
    * @param value the string value
    */
   public APropertyMetaData(String name, String value)
   {
      this.name = name;
      this.value = new StringValueMetaData(value);
   }

   /**
    * Create a new attribute meta data
    * 
    * @param name the name
    * @param value the value meta data
    */
   public APropertyMetaData(String name, IValueMetaData value)
   {
      this.name = name;
      this.value = value;
   }

   /**
    * Create a new property meta data
    * 
    * @param name the name
    * @param value the string value
    * @param type the type
    */
   public APropertyMetaData(String name, String value, String type)
   {
      this.name = name;
      StringValueMetaData svmd = new StringValueMetaData(value);
      svmd.setType(type);
      this.value = svmd;
   }

   public String getName()
   {
      return name;
   }

   /**
    * Set the name
    * 
    * @param name the name
    */
   @XmlAttribute
   public void setName(String name)
   {
      this.name = name;
   }

   public String getType()
   {
      if (value instanceof AbstractTypeMetaData)
      {
         return ((AbstractTypeMetaData)value).getType();
      }
      return null;
   }

   public boolean isPreInstantiate()
   {
      return preInstantiate;
   }

   @XmlAttribute(name="preinstantiate")
   public void setPreInstantiate(boolean preInstantiate)
   {
      this.preInstantiate = preInstantiate;
   }

   public IValueMetaData getValue()
   {
      return value;
   }

   public void setValue(IValueMetaData value)
   {
      this.value = value;
   }

   @XmlAnyElement
   @ManagementProperty(ignored = true)
   public void setValueObject(Object value)
   {
      if (value == null)
         setValue(null);
      else if (value instanceof IValueMetaData)
         setValue((IValueMetaData) value);
      else
         setValue(new AbstractValueMetaData(value));
   }

   @XmlValue
   @ManagementProperty(ignored = true)
   public void setValueString(String value)
   {
      if (value == null)
         setValue(null);
      else
      {
         IValueMetaData valueMetaData = getValue();
         if (valueMetaData instanceof StringValueMetaData)
         {
            ((StringValueMetaData) valueMetaData).setValue(value);
         }
         else
         {
            StringValueMetaData stringValue = new StringValueMetaData(value);
            stringValue.setType(getType());
            setValue(stringValue);
         }
      }
   }

   public String getPropertyType()
   {
      return type;
   }
   
   @XmlAttribute(name="class")
   @ManagementProperty(ignored = true)
   public void setPropertyType(String type)
   {
      this.type = type;
   }

   @XmlAttribute(name="replace")
   @ManagementProperty(ignored = true)
   public void setPropertyReplace(boolean replace)
   {
      this.replace = replace;
   }

   @XmlAttribute(name="trim")
   @ManagementProperty(ignored = true)
   public void setPropertyTrim(boolean trim)
   {
      this.trim = trim;
   }

   public void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(name);
      if (value != null)
         buffer.append(" value=").append(value);
   }

   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(name);
   }

   public APropertyMetaData clone()
   {
      APropertyMetaData clone = (APropertyMetaData)super.clone();
      return clone;
   }
}
