use 5.008_009;  # my_dirfd not available earlier
use strict;
use warnings;
use sigtrap qw(die normal-signals error-signals);
use Config;
use ExtUtils::Constant 'WriteConstants';
use ExtUtils::MakeMaker 'WriteMakefile';
use Fcntl;
use File::Spec;
use File::Temp 'mktemp';
use MIME::Base64 'decode_base64';

# Import only the non-XS part to get the constants for WriteConstants.
use lib File::Spec->catfile(File::Spec->curdir(), 'lib');
use POSIX::2008 '-noxs';

my $VERBOSE = exists $ENV{VERBOSE} ? $ENV{VERBOSE} : 0;

my $err2devnull = $VERBOSE ? '' : ' 2>'.File::Spec->devnull();
my $cfile = 'try.c';
my $tmp_ccout = mktemp('_tryXXXX');
my $tmp_exe = "${tmp_ccout}$Config{'_exe'}";

sub try_compile {
  my $what = shift;
  my $ccode = shift;

  my $lib =
    $what =~ /^dl/ ? '-ldl' :
    $what =~ /^(?:clock|timer)/ ? '-lrt' :
    $ccode =~ /<(?:complex|fenv|math)\.h>/ ? '-lm' : '';

  sysopen my $C, $cfile, O_WRONLY|O_CREAT|O_EXCL or die "$cfile: $!";
  syswrite $C, $ccode or die "Couldn't write to $cfile: $!";
  close $C;

  my $ccflags = $Config{ccflags};
  if ($what =~ /^sig[^n]/ && $Config{gccversion} && $Config{d_attribute_deprecated}) {
    $ccflags .= ' -Wno-deprecated-declarations';
  }
  my $cccmd = "$Config{cc} $ccflags -o $tmp_ccout $cfile $lib";
  $cccmd .= $err2devnull;

  print "$cccmd\n" if $VERBOSE;

  my $res = system $cccmd;
  my $ok = defined $res && $res == 0 && -s $tmp_exe && -x _;
  unlink $cfile, $tmp_ccout, $tmp_exe;

  return $ok;
}

unlink '2008.h', 'typemap', $cfile, $tmp_ccout, $tmp_exe;
sysopen my $DEF, '2008.h', O_WRONLY|O_CREAT|O_EXCL or die "2008.h: $!";

while (my $line = <DATA>) {
  next if $line =~ /^\s*(?:#.*)?$/;
  my ($what, $b64) = split ' ', $line;
  my $ccode = decode_base64($b64);
  my $ok = try_compile($what, $ccode);
  if ($ok) {
    my $define = 'PSX2008_HAS_' . uc $what;
    syswrite $DEF, "#define $define\n" or die "Couldn't write to 2008.h: $!";
  }
  print "$what: ".($ok ? 'yes' : 'no')."\n";
}
close DATA;
close $DEF;

WriteConstants(
  NAME => 'POSIX::2008',
  NAMES => \@POSIX::2008::_constants,
  PROXYSUBS => {croak_on_error => 1},
);

my %wmf_attributes = (
  NAME          => 'POSIX::2008',
  AUTHOR        => 'Carsten Gaebler',
  LICENSE       => 'unrestricted', # WTFPL
  VERSION_FROM  => 'lib/POSIX/2008.pm',
  ABSTRACT_FROM => 'lib/POSIX/2008.pod',
  MIN_PERL_VERSION => '5.008009',
  LIBS          => ['-lm -ldl -lrt'],
  DEFINE        => '',
  INC           => '-I.', # e.g., '-I. -I/usr/include/other'
  (
    $Config{gccversion} && $Config{d_attribute_deprecated} ?
    (CCFLAGS => $Config{ccflags} . ' -Wno-deprecated-declarations') : ()
  ),
  # Un-comment this if you add C files to link with later:
  # OBJECT      => '$(O_FILES)', # link all the C files too
  realclean     => {
    FILES => '2008.h const-c.inc const-xs.inc typemap try.c _try*'
  },
);
WriteMakefile(%wmf_attributes);

# Using T_NV is a bit ugly but how do you deal with e.g. an 8-byte off_t if
# ivsize is only 4 bytes?
my $iofft  = $Config{ivsize} < $Config{lseeksize} ? 'T_NV' : 'T_IV';
my $isizet = $Config{ivsize} < $Config{sizesize} ? 'T_NV' : 'T_IV';
(my $usizet = $isizet) =~ s/I/U/;

sysopen my $TMAP, 'typemap', O_WRONLY|O_CREAT|O_EXCL or die "typemap: $!";
print $TMAP <<EOTM;
Off_t     $iofft
time_t    $isizet
clock_t   T_IV
clockid_t T_IV
gid_t     T_UV
uid_t     T_UV
pid_t     T_IV
dev_t     T_UV
mode_t    T_UV
id_t      T_UV
int *     T_OPAQUEPTR
nl_catd   T_PTR
timer_t   T_PTR
SysRet0   T_SYSRET0
SysRet0uv   T_SYSRET0_UV
SysRetTrue  T_SYSRET_TRUE
FILE *    T_STDIO
psx_fd_t  T_PSX_FD

######
INPUT
T_PSX_FD
	\$var = _psx_fileno(aTHX_ \$arg);

OUTPUT
T_SYSRET0
	if (\$var != -1)
	  sv_setiv(\$arg, (IV)\$var);
T_SYSRET0_UV
	if (\$var != -1)
	  sv_setuv(\$arg, (UV)\$var);
T_SYSRET_TRUE
	if (\$var == 0)
	  sv_setpvn(\$arg, "0 but true", 10);
EOTM
close $TMAP or die "Couldn't write to typemap: $!";

END {
  unlink $cfile, $tmp_ccout, $tmp_exe;
}

__DATA__
complex_h I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
dlfcn_h I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
fnmatch_h I2luY2x1ZGUgPGZubWF0Y2guaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
libgen_h I2luY2x1ZGUgPGxpYmdlbi5oPgppbnQgbWFpbigpIHsgcmV0dXJuIDA7IH0K
nl_types_h I2luY2x1ZGUgPG5sX3R5cGVzLmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
poll_h I2luY2x1ZGUgPHBvbGwuaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
signal_h I2luY2x1ZGUgPHNpZ25hbC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIDA7IH0K
string_h I2luY2x1ZGUgPHN0cmluZy5oPgppbnQgbWFpbigpIHsgcmV0dXJuIDA7IH0K
strings_h I2luY2x1ZGUgPHN0cmluZ3MuaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
sys_poll_h I2luY2x1ZGUgPHN5cy9wb2xsLmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
sys_uio_h I2luY2x1ZGUgPHN5cy91aW8uaD4KaW50IG1haW4oKSB7IHJldHVybiAwOyB9Cg==
utmpx_h I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgeyByZXR1cm4gMDsgfQo=
a64l I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGxvbmcgbCA9IGE2NGwoYXJndlswXSk7CiAgcmV0dXJuIGwgJSAyOwp9Cg==
abort I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBpZiAoYXJnYyA+IDEpIGFib3J0KCk7CiAgcmV0dXJuIDA7Cn0K
abs I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGFicyhhcmdjKTsgfQo=
access I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gYWNjZXNzKGFyZ3ZbMF0sIFhfT0spOwp9Cg==
acos I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSk7CiAgcmV0dXJuIChpbnQpYWNvcyh4KTsKfQo=
acosh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpYWNvc2goeCk7Cn0K
alarm I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGFsYXJtKGFyZ2MpOyB9Cg==
asin I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSk7CiAgcmV0dXJuIChpbnQpYXNpbih4KTsKfQo=
asinh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSkqMTA7CiAgcmV0dXJuIChpbnQpYXNpbmgoeCk7Cn0K
atan I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSkqMTA7CiAgcmV0dXJuIChpbnQpYXRhbih4KTsKfQo=
atan2 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAxLjA7CiAgZG91YmxlIHkgPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSkqMTA7CiAgcmV0dXJuIChpbnQpYXRhbjIoeSwgeCk7Cn0K
atanh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAwLjkqKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpOwogIHJldHVybiAoaW50KWF0YW5oKHgpOwp9Cg==
atof I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClhdG9mKGFyZ3ZbMV0pOwp9Cg==
atoi I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGF0b2koYXJndlsxXSk7IH0K
atol I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpYXRvbChhcmd2WzFdKTsgfQo=
atoll I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gKGludClhdG9sbChhcmd2WzFdKTsgfQo=
basename I2luY2x1ZGUgPGxpYmdlbi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpiID0gYmFzZW5hbWUoYXJndlswXSk7CiAgcmV0dXJuIChpbnQpYlswXTsKfQo=
cabs I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYWJzKHopOwp9Cg==
cacos I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYWNvcyh6KTsKfQo=
cacosh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYWNvc2goeik7Cn0K
carg I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSkoKGFyZ2MgJSAzKSAtIDEpOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIC4xICogSTsKICByZXR1cm4gKGludCljYXJnKHopOwp9Cg==
casin I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYXNpbih6KTsKfQo=
casinh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYXNpbmgoeik7Cn0K
catan I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYXRhbih6KTsKfQo=
catanh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljYXRhbmgoeik7Cn0K
catclose I2luY2x1ZGUgPG5sX3R5cGVzLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gY2F0Y2xvc2UoKG5sX2NhdGQpYXJndik7IH0K
catgets I2luY2x1ZGUgPG5sX3R5cGVzLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKm1zZyA9IGNhdGdldHMoKG5sX2NhdGQpYXJndiwgMCwgMCwgIiIpOwogIHJldHVybiBtc2cgPyAwIDogLTE7Cn0K
catopen I2luY2x1ZGUgPG5sX3R5cGVzLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIG5sX2NhdGQgY2F0ZCA9IGNhdG9wZW4oYXJndlswXSwgMCk7CiAgcmV0dXJuIGNhdGQgPyAwIDogLTE7Cn0K
cbrt I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpY2JydCh4KTsKfQo=
ccos I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljY29zKHopOwp9Cg==
ccosh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljY29zaCh6KTsKfQo=
ceil I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKS8oZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpY2VpbCh4KTsKfQo=
cexp I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljZXhwKHopOwp9Cg==
chdir I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGNoZGlyKGFyZ3ZbMV0pOyB9Cg==
chmod I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gY2htb2QoYXJndlsxXSwgYXJnYyk7IH0K
chown I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGNob3duKGFyZ3ZbMV0sIGFyZ2MsIGFyZ2MpOyB9Cg==
cimag I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljaW1hZyh6KTsKfQo=
clock I2luY2x1ZGUgPHRpbWUuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiAoaW50KWNsb2NrKCk7IH0K
clock_getcpuclockid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjbG9ja2lkX3QgY2xvY2tpZDsKICByZXR1cm4gY2xvY2tfZ2V0Y3B1Y2xvY2tpZCgwLCAmY2xvY2tpZCk7Cn0K
clock_getres I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdGltZXNwZWMgcmVzOwogIHJldHVybiBjbG9ja19nZXRyZXMoQ0xPQ0tfUkVBTFRJTUUsICZyZXMpOwp9Cg==
clock_gettime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdGltZXNwZWMgcmVzOwogIHJldHVybiBjbG9ja19nZXR0aW1lKENMT0NLX1JFQUxUSU1FLCAmcmVzKTsKfQo=
clock_nanosleep I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsKICBzdHJ1Y3QgdGltZXNwZWMgcmVxID0gezB9OwogIHN0cnVjdCB0aW1lc3BlYyByZW07CiAgcmV0dXJuIGNsb2NrX25hbm9zbGVlcChDTE9DS19SRUFMVElNRSwgMCwgJnJlcSwgJnJlbSk7Cn0K
clock_settime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdGltZXNwZWMgcmVzID0geyAudHZfc2VjID0gMCwgLnR2X25zZWMgPSAwIH07CiAgcmV0dXJuIGNsb2NrX3NldHRpbWUoQ0xPQ0tfUkVBTFRJTUUsICZyZXMpOwp9Cg==
clog I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljbG9nKHopOwp9Cg==
close I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIGNsb3NlKDApOyB9Cg==
confstr I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyIGJ1ZlsxNl07CiAgcmV0dXJuIChpbnQpY29uZnN0cihfQ1NfUEFUSCwgYnVmLCBzaXplb2YoYnVmKSk7Cn0K
conj I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljb25qKHopOwp9Cg==
copysign I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgZG91YmxlIHkgPSAoZG91YmxlKWFyZ2MgLSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICByZXR1cm4gKGludCljb3B5c2lnbih4LCB5KTsKfQo=
cos I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpKGNvcyh4KSoxMCk7Cn0K
cosh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDUpIC0gMyk7CiAgcmV0dXJuIChpbnQpY29zaCh4KTsKfQo=
cpow I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIGEgPSAoZG91YmxlKWFyZ2M7CiAgZG91YmxlIGIgPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgY29tcGxleCB4ID0gYSArIGIgKiBJOwogIGRvdWJsZSBjb21wbGV4IHogPSBiICsgYSAqIEk7CiAgcmV0dXJuIChpbnQpY3Bvdyh4LCB6KTsKfQo=
cproj I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljcHJvaih6KTsKfQo=
creal I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljcmVhbCh6KTsKfQo=
creat I2luY2x1ZGUgPGZjbnRsLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gY3JlYXQoYXJndlsxXSwgMDYwMCk7IH0K
csin I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljc2luKHopOwp9Cg==
csinh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljc2luaCh6KTsKfQo=
csqrt I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljc3FydCh6KTsKfQo=
ctan I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljdGFuKHopOwp9Cg==
ctanh I2luY2x1ZGUgPGNvbXBsZXguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHJlID0gKGRvdWJsZSlhcmdjOwogIGRvdWJsZSBpbSA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSBjb21wbGV4IHogPSByZSArIGltICogSTsKICByZXR1cm4gKGludCljdGFuaCh6KTsKfQo=
dirname I2luY2x1ZGUgPGxpYmdlbi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyICpiID0gZGlybmFtZShhcmd2WzBdKTsKICByZXR1cm4gKGludCliWzBdOwp9Cg==
div I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBkaXZfdCBxID0gZGl2KGFyZ2MsIChpbnQpKGFyZ3ZbMF1bMF0pKTsKICByZXR1cm4gKGludClxLnF1b3Q7Cn0K
dlclose I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gZGxjbG9zZSgodm9pZCopYXJndlswXSk7IH0K
dlerror I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKmVyciA9IGRsZXJyb3IoKTsKICByZXR1cm4gZXJyID8gMSA6IDA7Cn0K
dlopen I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHZvaWQgKmhhbmRsZSA9IGRsb3Blbihhcmd2WzBdLCBSVExEX0xBWlkpOwogIHJldHVybiBoYW5kbGUgPyAwIDogMTsKfQo=
dlsym I2luY2x1ZGUgPGRsZmNuLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHZvaWQgKnN5bSA9IGRsc3ltKCh2b2lkKikwLCAiIik7CiAgcmV0dXJuIHN5bSA/IDAgOiAxOwp9Cg==
drand48 I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClkcmFuZDQ4KCk7Cn0K
endutxent I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgewogIGVuZHV0eGVudCgpOwogIHJldHVybiAwOwp9Cg==
erand48 I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICB1bnNpZ25lZCBzaG9ydCB4c3ViaVszXTsKICByZXR1cm4gKGludCllcmFuZDQ4KHhzdWJpKTsKfQo=
erf I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZXJmKHgpOwp9Cg==
erfc I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZXJmYyh4KTsKfQo=
execveat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKmNvbnN0IGVudnBbXSA9IHsgTlVMTCB9OwogIGNvbnN0IGNoYXIgKnBhdGggPSBhcmd2WzBdOwogIHdoaWxlICgqcGF0aCA9PSAnLycpIHBhdGgrKzsKICByZXR1cm4gZXhlY3ZlYXQoYXJnYywgcGF0aCwgYXJndiwgZW52cCwgMCk7Cn0K
exp I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZXhwKHgpOwp9Cg==
exp2 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZXhwMih4KTsKfQo=
expm1 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZXhwbTEoeCk7Cn0K
faccessat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gZmFjY2Vzc2F0KEFUX0ZEQ1dELCBhcmd2WzBdLCBYX09LLCAwKTsKfQo=
fchdir I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZmNoZGlyKC0xKTsgfQo=
fchmod I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KaW50IG1haW4oKSB7IHJldHVybiBmY2htb2QoLTEsIDApOyB9Cg==
fchmodat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBmY2htb2RhdChBVF9GRENXRCwgYXJndlsxXSwgMCwgQVRfU1lNTElOS19OT0ZPTExPVyk7Cn0K
fchown I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZmNob3duKC0xLCAtMSwgLTEpOyB9Cg==
fchownat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gZmNob3duYXQoQVRfRkRDV0QsIGFyZ3ZbMV0sIC0xLCAtMSwgQVRfU1lNTElOS19OT0ZPTExPVyk7Cn0K
fdatasync I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZmRhdGFzeW5jKDApOyB9Cg==
fdim I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpZmRpbSgoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7Cn0K
fdopen I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkaW8uaD4KaW50IG1haW4oKSB7IHJldHVybiBmZG9wZW4oMCwgInJiIikgPyAwIDogMTsgfQo=
fdopendir I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3R5cGVzLmg+CiNpbmNsdWRlIDxkaXJlbnQuaD4KaW50IG1haW4oKSB7IHJldHVybiBmZG9wZW5kaXIoMCkgPyAwIDogMTsgfQo=
feclearexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZWNsZWFyZXhjZXB0KGFyZ2MpOyB9Cg==
fegetround I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oKSB7IHJldHVybiBmZWdldHJvdW5kKCk7IH0K
feraiseexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZXJhaXNlZXhjZXB0KGFyZ2MpOyB9Cg==
fesetround I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZXNldHJvdW5kKGFyZ2MpOyB9Cg==
fetestexcept I2luY2x1ZGUgPGZlbnYuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmZXRlc3RleGNlcHQoYXJnYyk7IH0K
fexecve I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKmNvbnN0IGVudnBbXSA9IHsgTlVMTCB9OwogIHJldHVybiBmZXhlY3ZlKGFyZ2MsIGFyZ3YsIGVudnApOwp9Cg==
ffs I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5ncy5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGZmcyhhcmdjKTsgfQo=
ffsl I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gKGludClmZnNsKGFyZ2MpOyB9Cg==
ffsll I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gKGludClmZnNsbChhcmdjKTsgfQo=
floor I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKS8oZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpZmxvb3IoeCk7Cn0K
fma I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgZG91YmxlIHkgPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICBkb3VibGUgeiA9IChkb3VibGUpYXJnYzsKICByZXR1cm4gKGludClmbWEoeCwgeSwgeik7Cn0K
fmax I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpZm1heCgoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7Cn0K
fmin I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpZm1pbigoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJndlswXVswXSk7Cn0K
fmod I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIChpbnQpZm1vZCgoZG91YmxlKWFyZ3ZbMF1bMF0sIChkb3VibGUpYXJnYyk7Cn0K
fnmatch I2luY2x1ZGUgPGZubWF0Y2guaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmbm1hdGNoKCIqIiwgYXJndlswXSwgMCk7IH0K
fpathconf I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpZnBhdGhjb25mKGFyZ2MsIDApOyB9Cg==
fpclassify I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBmcGNsYXNzaWZ5KChkb3VibGUpYXJnYyk7IH0K
fstat I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKCkgewogIHN0cnVjdCBzdGF0IHN0OwogIHJldHVybiBmc3RhdCgwLCAmc3QpOwp9Cg==
fstatat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHJldHVybiBmc3RhdGF0KEFUX0ZEQ1dELCBhcmd2WzBdLCAmc3QsIDApOwp9Cg==
fstatvfs I2luY2x1ZGUgPHN5cy9zdGF0dmZzLmg+CmludCBtYWluKCkgewogIHN0cnVjdCBzdGF0dmZzIHN0OwogIHJldHVybiBmc3RhdHZmcygwLCAmc3QpOwp9Cg==
fsync I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZnN5bmMoMCk7IH0K
ftruncate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZnRydW5jYXRlKDAsIC0xKTsgfQo=
futimens I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KaW50IG1haW4oKSB7CiAgc3RydWN0IHRpbWVzcGVjIHRpbWVzWzJdID0gewogICAgeyAudHZfc2VjID0gMCwgLnR2X25zZWMgPSBVVElNRV9PTUlUIH0sCiAgICB7IC50dl9zZWMgPSAwLCAudHZfbnNlYyA9IFVUSU1FX09NSVQgfSwKICB9OwogIHJldHVybiBmdXRpbWVucygwLCB0aW1lcyk7Cn0K
getdate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdG0gKmQgPSBnZXRkYXRlKGFyZ3ZbMV0pOwogIHJldHVybiBkID8gKGludClkLT50bV9zZWMgOiAtMTsKfQo=
getdate_err I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIGdldGRhdGVfZXJyOyB9Cg==
getegid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIChpbnQpZ2V0ZWdpZCgpOyB9Cg==
geteuid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIChpbnQpZ2V0ZXVpZCgpOyB9Cg==
getgid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIChpbnQpZ2V0Z2lkKCk7IH0K
gethostid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gKGludClnZXRob3N0aWQoKTsgfQo=
gethostname I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgewogIGNoYXIgbmFtZVsyNTZdOwogIHJldHVybiBnZXRob3N0bmFtZShuYW1lLCBzaXplb2YobmFtZSkpOwp9Cg==
getitimer I2luY2x1ZGUgPHN5cy90aW1lLmg+CmludCBtYWluKCkgewogIHN0cnVjdCBpdGltZXJ2YWwgY3Vycl92YWx1ZTsKICByZXR1cm4gZ2V0aXRpbWVyKElUSU1FUl9SRUFMLCAmY3Vycl92YWx1ZSk7Cn0K
getpriority I2luY2x1ZGUgPHN5cy9yZXNvdXJjZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIGdldHByaW9yaXR5KFBSSU9fUFJPQ0VTUywgMCk7IH0K
getsid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gZ2V0c2lkKDApOyB9Cg==
getuid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIChpbnQpZ2V0dWlkKCk7IH0K
getutxent I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgewogIHN0cnVjdCB1dG1weCAqZW50ID0gZ2V0dXR4ZW50KCk7CiAgcmV0dXJuIGVudCA/IGVudC0+dXRfdHlwZSA6IC0xOwp9Cg==
getutxid I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgewogIHN0cnVjdCB1dG1weCB1dCA9IHswfTsKICBzdHJ1Y3QgdXRtcHggKmVudCA9IGdldHV0eGlkKCZ1dCk7CiAgcmV0dXJuIGVudCA/IGVudC0+dXRfdHlwZSA6IC0xOwp9Cg==
getutxline I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgewogIHN0cnVjdCB1dG1weCB1dCA9IHswfTsKICBzdHJ1Y3QgdXRtcHggKmVudCA9IGdldHV0eGxpbmUoJnV0KTsKICByZXR1cm4gZW50ID8gZW50LT51dF90eXBlIDogLTE7Cn0K
hypot I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHgsIHk7CiAgeCA9IChkb3VibGUpYXJnYzsKICB5ID0gKGRvdWJsZSkoYXJndlswXVswXSk7CiAgcmV0dXJuIChpbnQpaHlwb3QoeCwgeSk7Cn0K
ilogb I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBpbG9nYigoZG91YmxlKWFyZ2MpOyB9Cg==
isalnum I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNhbG51bShhcmdjKTsgfQo=
isalpha I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNhbHBoYShhcmdjKTsgfQo=
isascii I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8Y3R5cGUuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBpc2FzY2lpKGFyZ2MpOyB9Cg==
isatty I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIGlzYXR0eSgwKTsgfQo=
isblank I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8Y3R5cGUuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiBpc2JsYW5rKGFyZ2MpOyB9Cg==
iscntrl I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNjbnRybChhcmdjKTsgfQo=
isdigit I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNkaWdpdChhcmdjKTsgfQo=
isfinite I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzZmluaXRlKChkb3VibGUpYXJnYyk7IH0K
isgraph I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNncmFwaChhcmdjKTsgfQo=
isgreaterequal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzZ3JlYXRlcmVxdWFsKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
isinf I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzaW5mKChkb3VibGUpYXJnYyk7IH0K
isless I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzcygoZG91YmxlKWFyZ2MsIChkb3VibGUpYXJnYyk7IH0K
islessequal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzc2VxdWFsKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
islessgreater I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbGVzc2dyZWF0ZXIoKGRvdWJsZSlhcmdjLCAoZG91YmxlKWFyZ2MpOyB9Cg==
islower I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNsb3dlcihhcmdjKTsgfQo=
isnan I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbmFuKChkb3VibGUpYXJnYyk7IH0K
isnormal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzbm9ybWFsKChkb3VibGUpYXJnYyk7IH0K
isprint I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNwcmludChhcmdjKTsgfQo=
ispunct I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNwdW5jdChhcmdjKTsgfQo=
isspace I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXNzcGFjZShhcmdjKTsgfQo=
isunordered I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGlzdW5vcmRlcmVkKChkb3VibGUpYXJnYywgKGRvdWJsZSlhcmdjKTsgfQo=
isupper I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXN1cHBlcihhcmdjKTsgfQo=
isxdigit I2luY2x1ZGUgPGN0eXBlLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gaXN4ZGlnaXQoYXJnYyk7IH0K
j0 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpajAoeCk7Cn0K
j1 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpajEoeCk7Cn0K
jn I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICByZXR1cm4gKGludClqbihhcmdjLCB4KTsKfQo=
jrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHVuc2lnbmVkIHNob3J0IHhzdWJpWzNdOwogIHJldHVybiAoaW50KWpyYW5kNDgoeHN1YmkpOwp9Cg==
killpg I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgeyByZXR1cm4ga2lsbHBnKDAsIDE1KTsgfQo=
l64a I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKmE2NCA9IGw2NGEoKGxvbmcpYXJnYyk7CiAgcmV0dXJuIChpbnQpYTY0WzBdOwp9Cg==
labs I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIGxhYnMoYXJnYyk7IH0K
lchown I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBsY2hvd24oYXJndlswXSwgKHVpZF90KWFyZ2MsIChnaWRfdClhcmdjKTsKfQo=
ldexp I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICByZXR1cm4gKGludClsZGV4cCh4LCBhcmdjKTsKfQo=
ldiv I2luY2x1ZGUgPHN0ZGxpYi5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBsb25nIG51bSA9IChsb25nKWFyZ2M7CiAgbG9uZyBkZW5vbSA9IChsb25nKShhcmd2WzBdWzBdKTsKICBsZGl2X3QgcSA9IGxkaXYobnVtLCBkZW5vbSk7CiAgcmV0dXJuIChpbnQpcS5xdW90Owp9Cg==
lgamma I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbGdhbW1hKHgpOwp9Cg==
link I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbGluayhhcmd2WzBdLCBhcmd2WzBdKTsKfQo=
linkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbGlua2F0KEFUX0ZEQ1dELCBhcmd2WzBdLCBBVF9GRENXRCwgYXJndlswXSwgMCk7Cn0K
llabs I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gbGxhYnMoYXJnYyk7IH0K
lldiv I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGxvbmcgbG9uZyBudW0gPSAobG9uZyBsb25nKWFyZ2M7CiAgbG9uZyBsb25nIGRlbm9tID0gKGxvbmcgbG9uZykoYXJndlswXVswXSk7CiAgbGxkaXZfdCBxID0gbGxkaXYobnVtLCBkZW5vbSk7CiAgcmV0dXJuIChpbnQpcS5xdW90Owp9Cg==
llround I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClsbHJvdW5kKChkb3VibGUpYXJnYyk7Cn0K
log I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbG9nKHgpOwp9Cg==
log10 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbG9nMTAoeCk7Cn0K
log1p I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbG9nMXAoeCk7Cn0K
log2 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbG9nMih4KTsKfQo=
logb I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpbG9nYih4KTsKfQo=
lrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKCkgeyByZXR1cm4gKGludClscmFuZDQ4KCk7IH0K
lround I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpbHJvdW5kKChkb3VibGUpYXJnYyk7IH0K
lstat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IHN0YXQgc3Q7CiAgcmV0dXJuIGxzdGF0KGFyZ3ZbMF0sICZzdCk7Cn0K
mkdir I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewojaWYgZGVmaW5lZChfV0lOMzIpICYmICFkZWZpbmVkKF9fQ1lHV0lOX18pCiAgcmV0dXJuIG1rZGlyKGFyZ3ZbMF0pOwojZWxzZQogIHJldHVybiBta2Rpcihhcmd2WzBdLCAwNzU1KTsKI2VuZGlmCn0K
mkdirat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBta2RpcmF0KEFUX0ZEQ1dELCBhcmd2WzBdLCAwNzU1KTsKfQo=
mkdtemp I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgKmQgPSBta2R0ZW1wKGFyZ3ZbMF0pOwogIHJldHVybiBkID8gMCA6IC0xOwp9Cg==
mkfifo I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBta2ZpZm8oYXJndlswXSwgMDYwMCk7Cn0K
mkfifoat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBta2ZpZm9hdChBVF9GRENXRCwgYXJndlswXSwgMDYwMCk7Cn0K
mknod I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3N0YXQuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIG1rbm9kKGFyZ3ZbMF0sIFNfSUZJRk98MDYwMCwgMCk7Cn0K
mknodat I2luY2x1ZGUgPGZjbnRsLmg+CiNpbmNsdWRlIDxzeXMvc3RhdC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gbWtub2RhdChBVF9GRENXRCwgYXJndlswXSwgU19JRklGT3wwNjAwLCAwKTsKfQo=
mkstemp I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gbWtzdGVtcChhcmd2WzBdKTsgfQo=
mrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKCkgeyByZXR1cm4gKGludCltcmFuZDQ4KCk7IH0K
nanosleep I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdGltZXNwZWMgcmVxID0geyAudHZfc2VjID0gYXJnYywgLnR2X25zZWMgPSBhcmdjIH07CiAgc3RydWN0IHRpbWVzcGVjIHJlbTsKICByZXR1cm4gbmFub3NsZWVwKCZyZXEsICZyZW0pOwp9Cg==
nearbyint I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBkb3VibGUgeCA9IChkb3VibGUpYXJnYzsKICByZXR1cm4gKGludCluZWFyYnlpbnQoeCk7Cn0K
nextafter I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBkb3VibGUgeCwgeTsKICB4ID0gKGRvdWJsZSlhcmdjOwogIHkgPSB4ICsgMTsKICByZXR1cm4gKGludCluZXh0YWZ0ZXIoeCwgeSk7Cn0K
nexttoward I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBkb3VibGUgeCwgeTsKICB4ID0gKGRvdWJsZSlhcmdjOwogIHkgPSB4ICsgMTsKICByZXR1cm4gKGludCluZXh0dG93YXJkKHgsIHkpOwp9Cg==
nfds_t I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSAiMjAwOC5oIgojaWYgZGVmaW5lZChQU1gyMDA4X0hBU19QT0xMX0gpCiNpbmNsdWRlIDxwb2xsLmg+CiNlbGlmIGRlZmluZWQoUFNYMjAwOF9IQVNfU1lTX1BPTExfSCkKI2luY2x1ZGUgPHN5cy9wb2xsLmg+CiNlbmRpZgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBuZmRzX3QgbmZkcyA9IChuZmRzX3QpYXJnYzsKICByZXR1cm4gbmZkcyAlIDI7Cn0K
nice I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgeyByZXR1cm4gbmljZSgwKTsgfQo=
nrand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHVuc2lnbmVkIHNob3J0IHhzdWJpWzNdOwogIHJldHVybiAoaW50KW5yYW5kNDgoeHN1YmkpOwp9Cg==
open I2luY2x1ZGUgPGZjbnRsLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiBvcGVuKGFyZ3ZbMF0sIE9fUkRPTkxZKTsKfQo=
openat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIG9wZW5hdChBVF9GRENXRCwgYXJndlswXSwgT19SRE9OTFkpOwp9Cg==
openat2 I2luY2x1ZGUgPGZjbnRsLmg+CiNpbmNsdWRlIDxsaW51eC9vcGVuYXQyLmg+CiNpbmNsdWRlIDxzeXMvc3lzY2FsbC5oPgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBvcGVuX2hvdyBob3cgPSB7MH07CiAgcmV0dXJuIChpbnQpc3lzY2FsbChTWVNfb3BlbmF0MiwgQVRfRkRDV0QsIGFyZ3ZbMF0sICZob3csIHNpemVvZihob3cpKTsKfQo=
pathconf I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gKGludClwYXRoY29uZihhcmd2WzBdLCAwKTsKfQo=
poll I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSAiMjAwOC5oIgojaWYgZGVmaW5lZChQU1gyMDA4X0hBU19QT0xMX0gpCiNpbmNsdWRlIDxwb2xsLmg+CiNlbGlmIGRlZmluZWQoUFNYMjAwOF9IQVNfU1lTX1BPTExfSCkKI2luY2x1ZGUgPHN5cy9wb2xsLmg+CiNlbmRpZgppbnQgbWFpbigpIHsKICBzdHJ1Y3QgcG9sbGZkIGZkczsKICByZXR1cm4gcG9sbCgmZmRzLCAwLCAwKTsKfQo=
posix_fadvise I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHBvc2l4X2ZhZHZpc2UoYXJnYywgYXJnYywgYXJnYywgUE9TSVhfRkFEVl9OT1JNQUwpOwp9Cg==
posix_fallocate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgcmV0dXJuIHBvc2l4X2ZhbGxvY2F0ZShhcmdjLCBhcmdjLCBhcmdjKTsKfQo=
pread I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKCkgewogIGNoYXIgYnVmWzhdOwogIHJldHVybiAoaW50KXByZWFkKC0xLCBidWYsIHNpemVvZihidWYpLCAxKTsKfQo=
preadv I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgppbnQgbWFpbigpIHsKICBjaGFyIGJ1Zls4XTsKICBzdHJ1Y3QgaW92ZWMgaW92WzFdID0ge3tidWYsIHNpemVvZihidWYpfX07CiAgcmV0dXJuIChpbnQpcHJlYWR2KC0xLCBpb3YsIDEsIDEpOwp9Cg==
preadv2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgppbnQgbWFpbigpIHsKICBjaGFyIGJ1Zls4XTsKICBzdHJ1Y3QgaW92ZWMgaW92WzFdID0ge3tidWYsIHNpemVvZihidWYpfX07CiAgcmV0dXJuIChpbnQpcHJlYWR2MigtMSwgaW92LCAxLCAxLCAwKTsKfQo=
psignal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgewogIHBzaWduYWwoMCwgTlVMTCk7CiAgcmV0dXJuIDA7Cn0K
ptsname I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKCkgeyByZXR1cm4gcHRzbmFtZSgwKSA/IDAgOiAxOyB9Cg==
ptsname_r I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKCkgewogIGNoYXIgbmFtZVsyNTZdOwogIHJldHVybiBwdHNuYW1lX3IoMCwgbmFtZSwgc2l6ZW9mKG5hbWUpKTsKfQo=
pwrite I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiAoaW50KXB3cml0ZSgwLCBhcmd2WzBdLCAxLCAxKTsKfQo=
pwritev I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgaW92ZWMgaW92WzFdID0ge3thcmd2WzBdLCAxfX07CiAgcmV0dXJuIChpbnQpcHdyaXRldigwLCBpb3YsIDEsIDEpOwp9Cg==
pwritev2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3lzL3Vpby5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgaW92ZWMgaW92WzFdID0ge3thcmd2WzBdLCAxfX07CiAgcmV0dXJuIChpbnQpcHdyaXRldjIoMCwgaW92LCAxLCAxLCAwKTsKfQo=
raise I2luY2x1ZGUgPHNpZ25hbC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHJhaXNlKDE1KTsgfQo=
random I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKCkgeyByZXR1cm4gKGludClyYW5kb20oKTsgfQo=
read I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsKICBjaGFyIGJ1Zls4XTsKICByZXR1cm4gcmVhZCgtMSwgYnVmLCBzaXplb2YoYnVmKSk7Cn0K
readlink I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIGNoYXIgYnVmWzI1Nl07CiAgcmV0dXJuIChpbnQpcmVhZGxpbmsoYXJndlswXSwgYnVmLCBzaXplb2YoYnVmKSk7Cn0K
readlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBjaGFyIGJ1ZlsyNTZdOwogIHJldHVybiAoaW50KXJlYWRsaW5rYXQoQVRfRkRDV0QsIGFyZ3ZbMF0sIGJ1Ziwgc2l6ZW9mKGJ1ZikpOwp9Cg==
readv I2luY2x1ZGUgPHN5cy91aW8uaD4KaW50IG1haW4oKSB7CiAgY2hhciBidWZbOF07CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YnVmLCBzaXplb2YoYnVmKX19OwogIHJldHVybiAoaW50KXJlYWR2KC0xLCBpb3YsIDEpOwp9Cg==
realpath I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gcmVhbHBhdGgoYXJndlswXSwgTlVMTCkgPyAwIDogLTE7IH0K
remainder I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBkb3VibGUgeCA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSB5ID0gKGRvdWJsZSlhcmdjOwogIHJldHVybiAoaW50KXJlbWFpbmRlcih4LCB5KTsKfQo=
remove I2luY2x1ZGUgPHN0ZGlvLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gcmVtb3ZlKGFyZ3ZbMV0pOyB9Cg==
remquo I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBpbnQgaW50X3ZhbDsKICBkb3VibGUgeCA9IChkb3VibGUpKGFyZ3ZbMF1bMF0pOwogIGRvdWJsZSB5ID0gKGRvdWJsZSlhcmdjOwogIHJldHVybiAoaW50KXJlbXF1byh4LCB5LCAmaW50X3ZhbCk7Cn0K
rename I2luY2x1ZGUgPHN0ZGlvLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiByZW5hbWUoYXJndlswXSwgYXJndlswXSk7Cn0K
renameat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN0ZGlvLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiByZW5hbWVhdChBVF9GRENXRCwgYXJndlswXSwgQVRfRkRDV0QsIGFyZ3ZbMF0pOwp9Cg==
renameat2 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN0ZGlvLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHJldHVybiByZW5hbWVhdDIoQVRfRkRDV0QsIGFyZ3ZbMF0sIEFUX0ZEQ1dELCBhcmd2WzBdLCBSRU5BTUVfRVhDSEFOR0UpOwp9Cg==
rmdir I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICByZXR1cm4gcm1kaXIoYXJndlswXSk7Cn0K
round I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiAoaW50KXJvdW5kKChkb3VibGUpYXJnYyk7IH0K
scalbln I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiAoaW50KXNjYWxibG4oKGRvdWJsZSlhcmdjLCAobG9uZylhcmdjKTsgfQo=
scalbn I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7IHJldHVybiAoaW50KXNjYWxibigoZG91YmxlKWFyZ2MsIGFyZ2MpOyB9Cg==
seed48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHVuc2lnbmVkIHNob3J0IHNlZWQxNnZbM107CiAgdW5zaWduZWQgc2hvcnQgKnMgPSBzZWVkNDgoc2VlZDE2dik7CiAgcmV0dXJuIChpbnQpc1swXTsKfQo=
setegid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gc2V0ZWdpZCgoZ2lkX3QpYXJnYyk7IH0K
seteuid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gc2V0ZXVpZCgodWlkX3QpYXJnYyk7IH0K
setgid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHNldGdpZCgoZ2lkX3QpYXJnYyk7IH0K
setitimer I2luY2x1ZGUgPHN5cy90aW1lLmg+CmludCBtYWluKCkgewogIHN0cnVjdCBpdGltZXJ2YWwgb2xkOwogIHN0cnVjdCBpdGltZXJ2YWwgbmV3ID0gewogICAgLml0X2ludGVydmFsID0geyAudHZfc2VjID0gMSwgLnR2X3VzZWMgPSAwIH0sCiAgICAuaXRfdmFsdWUgPSB7IC50dl9zZWMgPSAwLCAudHZfdXNlYyA9IDEgfSwKICB9OwogIHJldHVybiBzZXRpdGltZXIoSVRJTUVSX1JFQUwsICZuZXcsICZvbGQpOwp9Cg==
setpriority I2luY2x1ZGUgPHN5cy9yZXNvdXJjZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHNldHByaW9yaXR5KFBSSU9fUFJPQ0VTUywgMCwgMCk7IH0K
setregid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gc2V0cmVnaWQoKGdpZF90KWFyZ2MsIChnaWRfdClhcmdjKTsgfQo=
setreuid I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gc2V0cmV1aWQoKHVpZF90KWFyZ2MsICh1aWRfdClhcmdjKTsgfQo=
setsid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpc2V0c2lkKCk7IH0K
setuid I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHNldHVpZCgodWlkX3QpYXJnYyk7IH0K
setutxent I2luY2x1ZGUgPHV0bXB4Lmg+CmludCBtYWluKCkgewogIHNldHV0eGVudCgpOwogIHJldHVybiAwOwp9Cg==
sighold I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgeyByZXR1cm4gc2lnaG9sZCgxNSk7IH0K
sigignore I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgeyByZXR1cm4gc2lnaWdub3JlKDE1KTsgfQo=
signbit I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8bWF0aC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHNpZ25iaXQoKGRvdWJsZSlhcmdjKTsgfQo=
sigpause I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgeyByZXR1cm4gc2lncGF1c2UoMTUpOyB9Cg==
sigrelse I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c2lnbmFsLmg+CmludCBtYWluKCkgeyByZXR1cm4gc2lncmVsc2UoMTUpOyB9Cg==
sin I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpc2luKHgpOwp9Cg==
sinh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDUpIC0gMyk7CiAgcmV0dXJuIChpbnQpc2luaCh4KTsKfQo=
srand48 I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHNyYW5kNDgoKGxvbmcpYXJnYyk7CiAgcmV0dXJuIDA7Cn0K
srandom I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RkbGliLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHNyYW5kb20oYXJnYyk7CiAgcmV0dXJuIDA7Cn0K
st_atim I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHN0YXQoYXJndlswXSwgJnN0KTsKICByZXR1cm4gKGludClzdC5zdF9hdGltLnR2X25zZWM7Cn0K
st_atimensec I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHN0YXQoYXJndlswXSwgJnN0KTsKICByZXR1cm4gKGludClzdC5zdF9hdGltZW5zZWM7Cn0K
st_ctime I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHN0YXQoYXJndlswXSwgJnN0KTsKICByZXR1cm4gKGludClzdC5zdF9jdGltZTsKfQo=
stat I2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCBzdGF0IHN0OwogIHJldHVybiBzdGF0KGFyZ3ZbMF0sICZzdCk7Cn0K
statvfs I2luY2x1ZGUgPHN5cy9zdGF0dmZzLmg+CmludCBtYWluKCkgewogIHN0cnVjdCBzdGF0dmZzIHN0OwogIHJldHVybiBzdGF0dmZzKCIvIiwgJnN0KTsKfQo=
strnlen I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gKGludClzdHJubGVuKGFyZ3ZbMF0sIDUpOyB9Cg==
strptime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsKICBzdHJ1Y3QgdG0gdG07CiAgY2hhciAqcCA9IHN0cnB0aW1lKGFyZ3ZbMV0sICIlVCIsICZ0bSk7CiAgcmV0dXJuIHAgPyB0bS50bV9zZWMgOiAtMTsKfQo=
strsignal I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8c3RyaW5nLmg+CmludCBtYWluKCkgewogICh2b2lkKXN0cnNpZ25hbCgwKTsKICByZXR1cm4gMDsKfQo=
symlink I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gc3ltbGluayhhcmd2WzBdLCBhcmd2WzBdKTsgfQo=
symlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHN5bWxpbmthdChhcmd2WzBdLCBBVF9GRENXRCwgYXJndlswXSk7IH0K
sync I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsKICBzeW5jKCk7CiAgcmV0dXJuIDA7Cn0K
sysconf I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpc3lzY29uZihhcmdjKTsgfQo=
tan I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpdGFuKHgpOwp9Cg==
tanh I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKSgoYXJnYyAlIDMpIC0gMSk7CiAgcmV0dXJuIChpbnQpKHRhbmgoeCkqMTApOwp9Cg==
tgamma I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpdGdhbW1hKHgpOwp9Cg==
timer_create I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsKICB0aW1lcl90IHRpbWVyX2lkOwogIHJldHVybiB0aW1lcl9jcmVhdGUoQ0xPQ0tfUkVBTFRJTUUsIE5VTEwsICZ0aW1lcl9pZCk7Cn0K
timer_delete I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHRpbWVyX2RlbGV0ZSgodGltZXJfdCkwKTsgfQo=
timer_getoverrun I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHRpbWVyX2dldG92ZXJydW4oKHRpbWVyX3QpMCk7IH0K
timer_gettime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsKICBzdHJ1Y3QgaXRpbWVyc3BlYyBjdXJyX3ZhbHVlOwogIHJldHVybiB0aW1lcl9nZXR0aW1lKCh0aW1lcl90KTAsICZjdXJyX3ZhbHVlKTsKfQo=
timer_settime I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dGltZS5oPgppbnQgbWFpbigpIHsKICBzdHJ1Y3QgaXRpbWVyc3BlYyBuZXdfdmFsdWUgPSB7MH07CiAgc3RydWN0IGl0aW1lcnNwZWMgb2xkX3ZhbHVlID0gezB9OwogIHJldHVybiB0aW1lcl9zZXR0aW1lKCh0aW1lcl90KTAsIDAsICZuZXdfdmFsdWUsICZvbGRfdmFsdWUpOwp9Cg==
trunc I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2MgLyAoZG91YmxlKShhcmd2WzBdWzBdKTsKICByZXR1cm4gKGludCl0cnVuYyh4KTsKfQo=
truncate I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8dW5pc3RkLmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgeyByZXR1cm4gdHJ1bmNhdGUoYXJndlsxXSwgLTEpOyB9Cg==
ttyname I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsgcmV0dXJuIHR0eW5hbWUoMCkgPyAwIDogLTE7IH0K
ttyname_r I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbigpIHsKICBjaGFyIG5hbWVbMjU2XTsKICByZXR1cm4gdHR5bmFtZV9yKDAsIG5hbWUsIHNpemVvZihuYW1lKSk7Cn0K
unlink I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHVubGluayhhcmd2WzFdKTsgfQo=
unlinkat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIHVubGlua2F0KEFUX0ZEQ1dELCBhcmd2WzFdLCAwKTsgfQo=
utimensat I2lmbmRlZiBfR05VX1NPVVJDRQojZGVmaW5lIF9HTlVfU09VUkNFCiNlbmRpZgojaW5jbHVkZSA8ZmNudGwuaD4KI2luY2x1ZGUgPHN5cy9zdGF0Lmg+CmludCBtYWluKGludCBhcmdjLCBjaGFyICoqYXJndikgewogIHN0cnVjdCB0aW1lc3BlYyB0aW1lc1syXSA9IHsKICAgIHsgLnR2X3NlYyA9IDAsIC50dl9uc2VjID0gVVRJTUVfT01JVCB9LAogICAgeyAudHZfc2VjID0gMCwgLnR2X25zZWMgPSBVVElNRV9PTUlUIH0sCiAgfTsKICByZXR1cm4gdXRpbWVuc2F0KEFUX0ZEQ1dELCBhcmd2WzBdLCB0aW1lcywgQVRfU1lNTElOS19OT0ZPTExPVyk7Cn0K
write I2luY2x1ZGUgPHVuaXN0ZC5oPgppbnQgbWFpbihpbnQgYXJnYywgY2hhciAqKmFyZ3YpIHsgcmV0dXJuIChpbnQpd3JpdGUoMCwgYXJndlswXSwgMSk7IH0K
writev I2luY2x1ZGUgPHN5cy91aW8uaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgc3RydWN0IGlvdmVjIGlvdlsxXSA9IHt7YXJndlswXSwgMX19OwogIHJldHVybiAoaW50KXdyaXRldigwLCBpb3YsIDEpOwp9Cg==
y0 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpeTAoeCk7Cn0K
y1 I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKWFyZ2M7CiAgcmV0dXJuIChpbnQpeTEoeCk7Cn0K
yn I2luY2x1ZGUgPG1hdGguaD4KaW50IG1haW4oaW50IGFyZ2MsIGNoYXIgKiphcmd2KSB7CiAgZG91YmxlIHggPSAoZG91YmxlKShhcmd2WzBdWzBdKTsKICByZXR1cm4gKGludCl5bihhcmdjLCB4KTsKfQo=
