/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      tctables.cpp
 * \brief     Thermocouple lookup table class
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: tctables.cpp,v 1.1 2007/05/14 12:36:04 douglas Exp $
 *
 *************************************************************************** */

//  Class which stores a vector of thermocouple data tables
//  to make conversion from microvolts to degrees C easier

#include <math.h>
#include <assert.h>

#include "tctables.h"


///////////////////////////////////////////////////////////////
//
// Destructor
//
///////////////////////////////////////////////////////////////

ThermocoupleTables::~ThermocoupleTables()
  {
  // ensure we clean up fully
  m_ClearTables();
  }

///////////////////////////////////////////////////////////////
//
// Add a thermocouple table to the vector along with all
//  appropriate info
//
// NOTE: TCTypes are not case sensitive
//
///////////////////////////////////////////////////////////////

void ThermocoupleTables::AddTable ( char    TCType, 
                                    long *  pTCData_uV, 
                                    float   flMinTemp, 
                                    float   flMaxTemp,
                                    float   flTempInterval)
  {
  tTCDataTable *  pTCDataRecord;

  pTCDataRecord = new tTCDataTable;

  pTCDataRecord->TCType       = toupper(TCType);
  // we can store a pointer to this data since we know it will be
  //  valid until the dll is unloaded (this reference will then
  //  be destroyed by the destructor)
  pTCDataRecord->pTCData      = pTCData_uV;
  pTCDataRecord->MinTemp      = flMinTemp;
  pTCDataRecord->MaxTemp      = flMaxTemp;
  pTCDataRecord->TempInterval = flTempInterval;

  // push a pointer to our record onto the vector
  this->m_vecTCTable.push_back(pTCDataRecord);
  }

///////////////////////////////////////////////////////////////
//
// Convert a millivolt reading into degrees centigrade
//
///////////////////////////////////////////////////////////////

bool ThermocoupleTables::ConvertTemp (  char    tctype, 
                                        float   millivolts, 
                                        float * centigrade,
                                        bool  * overflow)
  {
  int   iTable;
  int   iTableCount;
  int   iLength;
  int   iLowerValue;
  int   iFirstGuess;
  float flInterpolationRatio;
  float flMicrovolts;
  bool  fTCTypeFound;
  std::vector<tTCDataTable *>::iterator iterTable;
  bool  (*CompareUpperTemp)(float, float);
  bool  (*CompareLowerTemp)(float, float);
  int   (*StepUp)(int);
  int   (*StepDown)(int);



  // find the record relating to the tctype
  iTableCount = (int)this->m_vecTCTable.size();
  iterTable = this->m_vecTCTable.begin();
  for (iTable = 0, fTCTypeFound = false; iTable < iTableCount; ++iTable, ++iterTable)
    {
    if ((*iterTable)->TCType == toupper(tctype))
      {
      fTCTypeFound = true;
      break;
      }
    }
  if (!fTCTypeFound)
    return false;

  // The data is stored in Microvolts, so convert
  flMicrovolts = millivolts * 1000.0f;

  iLength = (int)((((*iterTable)->MaxTemp - (*iterTable)->MinTemp) 
    / (*iterTable)->TempInterval) + 1);

  // check whether the microvolt scale's coefficient is positive (TCs) or negative (CJC)
  if ((*iterTable)->pTCData[iLength - 1] > (*iterTable)->pTCData[0])
    {
    CompareUpperTemp = GreaterThan;
    CompareLowerTemp = LessThan;
    StepUp = Increment;
    StepDown = Decrement;
    }
  else
    {
    CompareUpperTemp = LessThan;
    CompareLowerTemp = GreaterThan;
    StepUp = Decrement;
    StepDown = Increment;
    }

  // Check the reading is within limits
  //  if its outside of limits, simply saturate the output high or low
  if (CompareUpperTemp(flMicrovolts, (float)(*iterTable)->pTCData[iLength - 1]))
    {
    *centigrade = (*iterTable)->MaxTemp;
    if (overflow)
      *overflow = true;
    return true;
    }
  else if (CompareLowerTemp(flMicrovolts, (float)(*iterTable)->pTCData[0]))
    {
    *centigrade = (*iterTable)->MinTemp;
    if (overflow)
      *overflow = true;
    return true;
    }
  else
    {
    if (overflow)
      *overflow = false;
    }

  // Have a guess at which element of the array we should be looking at (faster)
  iFirstGuess = (int)(iLength * ((flMicrovolts - (*iterTable)->pTCData[0]) / ((*iterTable)->pTCData[iLength-1] - (*iterTable)->pTCData[0])));

  // traverse the array in the direction of increasing magnitude
  //
  for (iLowerValue = iFirstGuess; (iLowerValue < iLength) && (iLowerValue >= 0); iLowerValue = StepUp(iLowerValue))
    {
    if ((float)(*iterTable)->pTCData[iLowerValue] > flMicrovolts)
      break;
    }
  assert ((iLowerValue >= 0) && (iLowerValue < iLength));

  // traverse the array in the direction of decreasing magnitude
  //  (we want the closest match that is lower than microvolts)
  //
  for (; (iLowerValue < iLength) && (iLowerValue >= 0); iLowerValue = StepDown(iLowerValue))
    {
    if ((float)(*iterTable)->pTCData[iLowerValue] < flMicrovolts)
      break;
    }
  assert ((iLowerValue >= 0) && (iLowerValue < iLength));

  //////////////////////////////////////////////////////////////////////////////
  // Now convert the reading (assume linear interpolation)

  // StepUp(iLowerValue) will be higher than 'flMicrovolts'
  flInterpolationRatio = (float)(flMicrovolts - (*iterTable)->pTCData[iLowerValue]) / 
    ((*iterTable)->pTCData[StepUp(iLowerValue)] - (*iterTable)->pTCData[iLowerValue]);

  if ((*iterTable)->pTCData[iLength - 1] > (*iterTable)->pTCData[0])
    {
	//for +ve temp coeffs (TCs)
  *centigrade = ((flInterpolationRatio * (*iterTable)->TempInterval) + 
    ((iLowerValue * (*iterTable)->TempInterval) + (*iterTable)->MinTemp));
		}
	else
	{
	// for -ve coeffs (CJT)
  *centigrade = -((flInterpolationRatio * (*iterTable)->TempInterval) - 
    ((iLowerValue * (*iterTable)->TempInterval) + (*iterTable)->MinTemp));
	}																														
  return true;
  }

/////////////////////////////////////////////////////////////////////////
//
// 4 x compare functions for function pointers
//
/////////////////////////////////////////////////////////////////////////

static bool GreaterThan (float test1, float test2)
  {
  return (test1 > test2);
  }

static bool LessThan (float test1, float test2)
  {
  return (test1 < test2);
  }

static int Increment (int value)
  {
  return ++value;
  }

static int Decrement (int value)
  {
  return --value;
  }

///////////////////////////////////////////////////////////////
//
// Convert a centigrade reading into millivolts
//
///////////////////////////////////////////////////////////////

bool ThermocoupleTables::ConvertMillivolts (  char    tctype, 
                                              float   centigrade, 
                                              float * millivolts,
                                              bool  * overflow)
  {
  int   iTable;
  int   iTableCount;
  int   iLength;
  int   iLowerIndex;
  float flLowerTemp;
  float flInterpolationRatio;
  float flMicrovolts;
  float flIndex;
  bool  fTCTypeFound;
  std::vector<tTCDataTable *>::iterator iterTable;


  // find the record relating to the tctype
  iTableCount = (int)this->m_vecTCTable.size();
  iterTable = this->m_vecTCTable.begin();
  for (iTable = 0, fTCTypeFound = false; iTable < iTableCount; ++iTable, ++iterTable)
    {
    if ((*iterTable)->TCType == toupper(tctype))
      {
      fTCTypeFound = true;
      break;
      }
    }
  if (!fTCTypeFound)
    return false;

  iLength = (int)((((*iterTable)->MaxTemp - (*iterTable)->MinTemp) 
    / (*iterTable)->TempInterval) + 1);


  // Check the reading is within limits
  //  if its outside of limits, simply saturate the output high or low
  if (centigrade > (*iterTable)->MaxTemp)
    {
    *millivolts = (float)(*iterTable)->pTCData[iLength] / 1000.f;
    if (overflow)
      *overflow = true;
    return true;
    }
  else if (centigrade < (*iterTable)->MinTemp)
    {
    *millivolts = (float)(*iterTable)->pTCData[0] / 1000.0f;
    if (overflow)
      *overflow = true;
    return true;
    }
  else
    {
    if (overflow)
      *overflow = false;
    }


  // The temperature scale is linear so get the exact index as a float
  flIndex = (iLength-1) * ((centigrade - (*iterTable)->MinTemp) / ((*iterTable)->MaxTemp - (*iterTable)->MinTemp));
  
  // Round down the index
  flInterpolationRatio = fmodf(flIndex, 1.0f);
  iLowerIndex = (int)floor(flIndex);
  // Find the temperature at this index
  flLowerTemp = (*iterTable)->MinTemp + ((float)iLowerIndex * (*iterTable)->TempInterval);


  //////////////////////////////////////////////////////////////////////////////
  // Now convert the reading (assume linear interpolation)

  flMicrovolts = ((flInterpolationRatio * ((*iterTable)->pTCData[iLowerIndex+1] - (*iterTable)->pTCData[iLowerIndex]) + 
    (*iterTable)->pTCData[iLowerIndex]));


  // The data is stored in Microvolts, so convert & return it
  *millivolts = flMicrovolts / 1000.0f;

  return true;
  }

///////////////////////////////////////////////////////////////
//
// Destroy the tables / delete the memory allocated
//
///////////////////////////////////////////////////////////////

void ThermocoupleTables::m_ClearTables (void)
  {
  int iTable;
  int iTableCount;
  std::vector<tTCDataTable *>::iterator iterTable;

  iTableCount = (int)this->m_vecTCTable.size();
  iterTable = this->m_vecTCTable.begin();
  for (iTable = 0; iTable < iTableCount; ++iTable, ++iterTable)
    {
    // delete the table record
    delete (*iterTable);
    }

  this->m_vecTCTable.clear();
  }
