# Copyright 2013 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test whether or not a node is compatible with MAAS."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "TestOneNode",
    ]

import httplib
import json
import sys

from maastest import utils
from maastest.maas_enums import NODE_STATUS
import testresources
import testtools
from testtools.testcase import gather_details


class TestMAAS(testtools.TestCase, testresources.ResourcedTestCase):
    """Base class for testing machine compatibility with MAAS."""

    @classmethod
    def configure(cls, args, maas):
        """Set test parameters.

        These are class variables, but they are set only on ad-hoc subclasses,
        not on `TestMAAS` itself.  So this class stays in its original state.

        :param args: An arguments object for the program run, as returned by
            the arguments parser.
        :param maas: `MAASFixture` with a running MAAS.
        """
        assert cls != TestMAAS, "TestMAAS.configure called on TestMAAS class."
        cls.args = args
        cls.maas = maas
        cls.details = {}

    def shortDescription(self):
        """Overridable from `unittest`: Describe the ongoing test.

        By default this shows the first line of the test's docstring (if
        available), but this `TestCase` is basically our UI so we want full,
        user-friendly docstrings.
        """
        # Cribbed from unittest2.TestCase.  The difference is that we return
        # the whole docstring, instead of just the first line.
        # Future maintainers beware: the docstring may also be None.
        return self._testMethodDoc

    def unify_details(self):
        """Merge the details of the fixture into the details of the TestMAAS.
        """
        # Make sure that the MAAS logs are included in the test report.
        self.maas.collect_logs()
        gather_details(self.maas.kvm_fixture.getDetails(), self.getDetails())
        # This is a bit of a horrible hack, but it allows us to gather
        # up the details in the case of successful runs (which
        # testtools, rather frustratingly, doesn't support).
        gather_details(self.getDetails(), self.details)

    def setUp(self):
        super(TestMAAS, self).setUp()
        self.addCleanup(self.unify_details)

    def get_node_list(self, status):
        """Return the list of nodes with the given status."""
        # TODO: Move this to MAASFixture?
        uri = utils.get_uri('nodes/')
        response = self.maas.admin_maas_client.get(uri, op="list")
        self.assertEqual(
            httplib.OK, response.code, "Failed to get the node list.")
        nodes = json.loads(response.read())
        relevant_nodes = [node for node in nodes if node['status'] == status]
        return relevant_nodes


class TestOneNode(TestMAAS):
    """Test a single node for compatibility with MAAS."""

    def test_power_up_node(self):
        """Power up the node under test.

        If maas-test is running in interactive mode, this will ask the user
        to power up the node manually.  Otherwise it will use the provided
        power parameters to power up the node by running 'ipmipower' in the
        virtual machine..
        """
        if self.args.dry_run:
            self.skip("Dry run.")
        if self.args.interactive:
            self.power_up_node_interactive()
        else:
            self.power_up_node_noninteractive()

    def power_up_node_interactive(self):
        """Asks the user to power up the node manually."""
        print(
            "Power up the node under test and press enter to proceed "
            "with the testing.")
        sys.stdin.readline()

    def power_up_node_noninteractive(self):
        """Power up the node using the provided power parameters."""
        # Power-cycle the node.
        # TODO: Move these into MAASFixture?
        power_address = self.find_bmc_ip_address()
        self.maas.kvm_fixture.run_command([
            'ipmipower', '-h', power_address,
            '-W', 'opensesspriv',
            '-D', self.args.ipmi_driver,
            '-u', self.args.bmc_username,
            '-p', self.args.bmc_password, '--off'],
            check_call=True)
        self.maas.kvm_fixture.run_command([
            'ipmipower', '-h', power_address,
            '-W', 'opensesspriv',
            '-D', self.args.ipmi_driver,
            '-u', self.args.bmc_username,
            '-p', self.args.bmc_password, '--on'],
            check_call=True)

    def find_bmc_ip_address(self):
        """Returns the BMC IP address.

        This is done by scanning the network it the BMC MAC address was given
        or by simply returning the IP address of the BMC is it was passed.
        """
        if self.args.bmc_mac is not None:
            # The BMC's MAC address was passed: scan the network to find the
            # IP address of the node's BMC.
            for retry in utils.retries(delay=20, timeout=5 * 60):
                ip_scan = self.maas.kvm_fixture.get_ip_from_network_scan
                power_address = ip_scan(self.args.bmc_mac)
                if power_address is not None:
                    return power_address
        else:
            # The BMC IP address was passed: just return that.
            return self.args.bmc_ip
        if power_address is None:
            self.fail("Failed to get the IP address of the BMC.")

    def test_enlist_node(self):
        """Enlist node."""
        if self.args.dry_run:
            self.skip("Dry run.")
        for retry in utils.retries(delay=10, timeout=10 * 60):
            nb_enlisted_nodes = len(self.get_node_list(NODE_STATUS.DECLARED))
            if nb_enlisted_nodes == 1:
                return
        self.fail("Failed to enlist node.")

    def test_wait_node_down(self):
        """Wait for the node to go down."""
        if self.args.dry_run:
            self.skip("Dry run.")
        power_address = self.find_bmc_ip_address()
        for retry in utils.retries(delay=10, timeout=60):
            _, result, _ = self.maas.kvm_fixture.run_command([
                'ipmipower', '-h', power_address,
                '-W', 'opensesspriv',
                '-D', self.args.ipmi_driver,
                '-u', self.args.bmc_username,
                '-p', self.args.bmc_password, '--stat'],
                check_call=True)
            if result.strip() == "%s: off" % power_address:
                return
        self.fail("Node wasn't powered down after enlistment.")

    def test_commission_node(self):
        """Commission node."""
        if self.args.dry_run:
            self.skip("Dry run.")
        uri = utils.get_uri('nodes/')
        response = self.maas.admin_maas_client.post(uri, op="accept_all")
        self.assertEqual(
            httplib.OK, response.code, "Failed to accept node.")

        for retry in utils.retries(delay=10, timeout=10 * 60):
            nb_enlisted_nodes = len(self.get_node_list(NODE_STATUS.READY))
            if nb_enlisted_nodes == 1:
                return
        self.fail("Failed to commission node.")
