/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AbstractFiller.h"

#include <cstring>

#include <GTLCore/Math.h>
#include <GTLCore/Region.h>

#include "AbstractBrush.h"
#include "DrawingPoint.h"
#include "Path.h"
#include "DistanceInformation.h"

using namespace OpenRijn;

AbstractFiller::AbstractFiller() : d(0)
{
}

AbstractFiller::~AbstractFiller()
{
//   delete d;
}

void AbstractFiller::fillRegion(const GTLCore::RegionI& _region, float _opacity )
{
  for (int y = _region.top(); y <= _region.bottom(); ++y)
  {
    for (int x = _region.left(); x <= _region.right(); ++x)
    {
      fillAt(DrawingPoint(x, y, 1.0, 0.0), _opacity);
    }
  }
}

void AbstractFiller::fillRegion(const GTLCore::RegionF& _region, float _opacity )
{
  GTLCore::RegionI reg = _region.toRegionI();
  for (int y = reg.top(); y <= reg.bottom(); ++y)
  {
    float opacity_y = _opacity;
    if (y == reg.top()) {
      opacity_y = (1.0 - (_region.top() -y)) * _opacity;
    }
    if (y == reg.bottom()) {
      opacity_y = (1.0 - (y - _region.bottom())) * _opacity;
    }
    for (int x = reg.left(); x <= reg.right(); ++x)
    {
      float opacity_x = opacity_y;
      if (x == reg.left()) {
        opacity_x = (1.0 - (_region.left() -y)) * opacity_x;
      }
      if (y == reg.right()) {
        opacity_x = (1.0 - (y - _region.right())) * opacity_x;
      }
      fillAt(DrawingPoint(x, y, 1.0, 0.0), opacity_x);
    }
  }
}

class FillPathBrush : public AbstractBrush {
  public:
    FillPathBrush(float** img, int offsetX, int offsetY) : m_image(img), m_offsetX(offsetX), m_offsetY(offsetY) {}
    virtual ~FillPathBrush() {}
    virtual bool canPaintAt() const { return true; }
    virtual float drawPoint(AbstractCanvas* , const DrawingPoint& dp)
    {
      int x = dp.x;
      int y = dp.y;
      
      float dx = dp.x - x;
      float dy = dp.y - y;
      float idx = 1.0 - dx;
      float idy = 1.0 - dy;
      m_image[y][x] += idx * idy;
      m_image[y+1][x] += idx * dy;
      m_image[y+1][x+1] += dx * dy;
      m_image[y][x+1] += dx * idy;
      return 1.0;
    }

    void paintPath(const Path& _polyLine)
    {
      if( _polyLine.elementCount() == 0 ) return;
      DistanceInformation information;
      DrawingPoint lastPoint;
      std::size_t idx = 0;
      if( _polyLine.elementAt(0)->type == Path::MoveToElement)
      {
        lastPoint.x = _polyLine.elementAt(0)->point.x;
        lastPoint.y = _polyLine.elementAt(0)->point.y;
        ++idx;
      }
      DrawingPoint firstPoint = lastPoint;
      for(; idx < _polyLine.elementCount(); ++idx)
      {
          const Path::Element* elt = _polyLine.elementAt(idx);
          switch(elt->type)
          {
            case Path::CurveToElement:
            {
              const Path::CurveElement* celt = static_cast<const Path::CurveElement*>(elt);
              information = drawCurve( 0, lastPoint, celt->x1, celt->y1, celt->x2, celt->y2, elt->point, information);
              break;
            }
            case Path::MoveToElement:
            case Path::LineToElement:
              information = drawLine( 0, lastPoint, elt->point, information);
              break;
          }
          lastPoint = elt->point;
      }
      drawLine(0, lastPoint, firstPoint, information); // Close the loop
    }
  private:
    float** m_image;
    int m_offsetX, m_offsetY;
};

void AbstractFiller::fillPath(const Path& _polyLine, float _opacity )
{
  GTLCore::RegionI reg = _polyLine.boundingRegion().toRegionI();
  float** m_image = new float*[reg.rows()];
  for(int y = 0; y < reg.rows(); ++y)
  {
    m_image[y] = new float[reg.columns()];
    memset(m_image[y], 0, sizeof(float) * reg.columns());
  }
  FillPathBrush fpb(m_image, reg.left(), reg.top());
  fpb.paintPath(_polyLine);
  for(int y = 0; y < reg.rows(); ++y)
  {
    float count = 0.0;
    for(int x = 0; x < reg.columns(); ++x)
    {
      count += m_image[y][x];
      int countI = round(count);
      float d = std::fabs(countI - count);
      if( d < 0.01 )
      {
        if( (countI & 1) == 1)
        {
          fillAt(DrawingPoint(x, y, 1.0, 0.0));
        }
      } else {
        fillAt(DrawingPoint(x, y, 1.0, 0.0), d);
      }
      
    }
  }
}
