/*
 *  Copyright (c) 2009-2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Source.h"

using namespace OpenShiva;

#include <GTLCore/Debug.h>

#include "LightParser_p.h"
#include <GTLFragment/Source_p.h>
#include <GTLFragment/Lexer_p.h>

struct Source::Private : public GTLFragment::Source::Private
{
  Private() : type( InvalidSource ) {}
  Private(const Private& _rhs) : GTLFragment::Source::Private(_rhs), type( InvalidSource ) {}
  
  Private* clone() { return new Private(*this); }
  void specificUpdate(const GTLCore::String& _source, GTLCore::String& name);

  SourceType type;
  ImageType outputImageType;
  std::vector<ImageType> inputImageTypes;
};

void Source::Private::specificUpdate(const GTLCore::String& _source, GTLCore::String& name)
{
  std::istringstream iss(_source);
  GTLFragment::Lexer lexer( &iss, GTLFragment::Library::SHIVA_LIBRARY );
  LightParser parser(&lexer);
  parser.parse();
  name = parser.name();
  type = parser.sourceType();
  outputImageType = parser.outputImageType();
  inputImageTypes = parser.inputImageTypes();
  if( type != Library and outputImageType != InvalidImage )
  {
    switch( inputImageTypes.size() )
    {
      case 0:
        type = GeneratorKernel;
        break;
      case 1:
        type = FilterKernel;
        break;
      default:
        type = CompositionKernel;
        break;
    }
  }
}


// ---- Library::Private ---- //

const Source::Private* Source::d_func() const
{
  return static_cast<const Source::Private*>(d);
}

Source::Source() : GTLFragment::Source(d = new Private)
{
}

GTL_INHERITED_SHARED_DATA(Source, GTLFragment::Source)

Source::Source(const GTLFragment::Source& _src): GTLFragment::Source(_src)
{

}

Source::SourceType Source::sourceType() const
{
  d->update();
  return d_func()->type;
}

Source::ImageType Source::outputImageType() const
{
  d->update();
  return d_func()->outputImageType;
}

Source::ImageType Source::inputImageType(int idx) const
{
  d->update();
  if (std::size_t(idx) < d_func()->inputImageTypes.size())
  {
    return d_func()->inputImageTypes[idx];
  } else {
    return OpenShiva::Source::InvalidImage;
  }
}

int Source::countInputImages() const
{
  d->update();
  return d_func()->inputImageTypes.size();
}

