///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __CHANNEL_COLUMN_MAPPING_EDITOR_H
#define __CHANNEL_COLUMN_MAPPING_EDITOR_H

#include <core/Core.h>
#include <atomviz/AtomViz.h>
#include "ChannelColumnMapping.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

/**
 * \brief A widget that lets the user edit a ChannelColumnMapping.
 *
 * This widget class can be placed into a dialog box.
 *
 * \sa ChannelColumnMapping
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT ChannelColumnMappingEditor : public QWidget
{
public:

	/// \brief Initializes the widget.
	ChannelColumnMappingEditor(QWidget* parent = 0);

	/// \brief Fills the editor with the given mapping.
	/// \param mapping The data channel to column mapping that should be loaded into the editor.
	/// \param atomsObj The AtomsObject that should be exported. It is needed by the editor to
	///                 determine the available DataChannel objects that can be exported.
	void setMapping(const ChannelColumnMapping& mapping, AtomsObject* atomsObj);

	/// \brief Returns the current contents of the editor.
	/// \throws Exception if the current mapping in the editor is not valid.
	ChannelColumnMapping mapping() const;

protected:

	/// Makes sure that the table contains exactly one empty row at the end.
	void ensureEmptyRowAtEnd();

	/// Updates the labels of the vertical header.
	void updateHeaderLabels();

protected Q_SLOTS:

	/// Updates the entries in the "Presets" menu.
	void updatePresetMenu();

	/// Loads a preset.
	void onLoadPreset();

	/// Saves a preset under a new name.
	void onSavePresetAs();

	/// Saves a preset under an existing name.
	void onSavePreset();

	/// Deletes a preset.
	void onDeletePreset();

	/// Is called when the user pressed the "Output All Channels" button.
	void onOutputAllChannels();

private:

	class ChannelNameItemDelegate : public QItemDelegate {
	public:
		ChannelColumnMappingEditor* owner;
		virtual QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index) const {
			QComboBox* box = new QComboBox(parent);
			box->addItem("", 0);
			// Always add the special atom index channel.
			box->addItem(DataChannel::standardChannelName(DataChannel::AtomIndexChannel), DataChannel::AtomIndexChannel);
			Q_FOREACH(DataChannel* channel, owner->atoms->dataChannels()) {
				if(channel->id() != DataChannel::AtomIndexChannel && channel->type() != QMetaType::Void)
					box->addItem(channel->name(), channel->id());
			}
			return box;
		}
		virtual void setEditorData(QWidget* editor, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			int channelId = index.model()->data(index, Qt::UserRole).toInt();
			if(channelId < 0)
				box->setCurrentIndex(box->findData(channelId));
			else
				box->setCurrentIndex(box->findText(index.model()->data(index).toString()));
		}
		virtual void setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			QString newName = box->currentText();
			int newId = box->itemData(box->currentIndex()).toInt();
			model->setData(index, newName);
			model->setData(index, newId, Qt::UserRole);
			if(box->currentIndex() > 0 && newId != DataChannel::AtomIndexChannel) {
				int vectorComponent = index.sibling(index.row(), 1).data(Qt::UserRole).toInt();
				OVITO_ASSERT(vectorComponent >= 0);
				Q_FOREACH(DataChannel* channel, owner->atoms->dataChannels()) {
					if(channel->id() == newId && channel->name() == newName) {
						vectorComponent = min(vectorComponent, (int)channel->componentCount()-1);
						QString componentName;
						if(channel->componentNames().size() > vectorComponent) componentName = channel->componentNames()[vectorComponent];
						model->setData(index.sibling(index.row(), 1), vectorComponent, Qt::UserRole);
						model->setData(index.sibling(index.row(), 1), componentName);
						break;
					}
				}
			}
			else {
				model->setData(index.sibling(index.row(), 1), 0, Qt::UserRole);
				model->setData(index.sibling(index.row(), 1), "");
			}
			owner->ensureEmptyRowAtEnd();
		}
	};

	class VectorComponentItemDelegate : public QItemDelegate {
	public:
		ChannelColumnMappingEditor* owner;
		virtual QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option, const QModelIndex& index ) const {
			QComboBox* box = new QComboBox(parent);
			return box;
		}
		virtual void setEditorData(QWidget* editor, const QModelIndex& index) const {
			int value = index.model()->data(index, Qt::UserRole).toInt();
			QComboBox* box = static_cast<QComboBox*>(editor);
			box->clear();
			DataChannel::DataChannelIdentifier dataChannelId = (DataChannel::DataChannelIdentifier)index.sibling(index.row(), 0).data(Qt::UserRole).toInt();
			DataChannel* channel;
			if(dataChannelId != DataChannel::UserDataChannel)
				channel = owner->atoms->getStandardDataChannel(dataChannelId);
			else
				channel = owner->atoms->findDataChannelByName(index.sibling(index.row(), 0).data().toString());
			if(channel && channel->type() != QMetaType::Void && channel->componentCount() > 1) {
				Q_FOREACH(QString name, channel->componentNames())
					box->addItem(name);
				box->setCurrentIndex(value);
				box->setEnabled(true);
			}
			else box->setEnabled(false);
		}
		virtual void setModelData(QWidget* editor, QAbstractItemModel* model, const QModelIndex& index) const {
			QComboBox* box = static_cast<QComboBox*>(editor);
			if(box->currentIndex() >= 0) {
				model->setData(index, box->currentIndex(), Qt::UserRole);
				model->setData(index, box->currentText());
			}
			else {
				model->setData(index, 0, Qt::UserRole);
				model->setData(index, "");
			}
			owner->ensureEmptyRowAtEnd();
		}
	};

private:

	/// The main table widget that entries for each data column of the input file.
	QTableWidget* tableWidget;

	/// This menu is shown when the "Presets" menu button is pressed.
	QMenu presetMenu;

	/// The AtomsObject that is going to be exported.
	intrusive_ptr<AtomsObject> atoms;

	ChannelNameItemDelegate nameItemDelegate;
	VectorComponentItemDelegate vectorComponentItemDelegate;

	Q_OBJECT
};

};	// End of namespace AtomViz

#endif // __CHANNEL_COLUMN_MAPPING_EDITOR_H
