///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/input/RotationMode.h>
#include <core/viewport/input/XFormManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

/******************************************************************************
* Is called when the xform operation begins.
******************************************************************************/
void RotationMode::startXForm()
{
	// Get the transformation center for the first node in the selection set.
	if(DATASET_MANAGER.currentSelection()->empty()) return;
	SceneNode* masterNode = DATASET_MANAGER.currentSelection()->node(0);
	transformationCenter = XFORM_MANAGER.getTransformationCenter(masterNode);
}

/******************************************************************************
* Is called during the operation.
******************************************************************************/
void RotationMode::doXForm()
{
	FloatType angle1 = (FloatType)(currentPoint.y() - clickPoint.y()) / 100.0;
	FloatType angle2 = (FloatType)(currentPoint.x() - clickPoint.x()) / 100.0;

	// Snap angles.
	angle1 = SNAPPING_MANAGER.snapAngle(angle1);
	angle2 = SNAPPING_MANAGER.snapAngle(angle2);

	// Constrain rotation to z-axis.
	rot = Rotation(Vector3(0,0,1), angle1);

	// Apply transformation to selected nodes.
	applyXForm(DATASET_MANAGER.currentSelection()->nodes(), 1.0);
}

/******************************************************************************
* Applies the current transformation to the given nodes.
******************************************************************************/
void RotationMode::applyXForm(const QVector<SceneNode*>& nodeSet, FloatType multiplier)
{
	TimeTicks time = ANIM_MANAGER.time();
	
	Q_FOREACH(SceneNode* node, nodeSet) {
		
		CHECK_OBJECT_POINTER(node->transformationController());
		
		// Get transformation system.
		AffineTransformation transformSystem;
		XFORM_MANAGER.getTransformationSystem(node, transformSystem);

		if(XFORM_MANAGER.centerMode() == XFormManager::SELECTION_CENTER) {
            transformSystem.setTranslation(transformationCenter - ORIGIN);
		}

		// Make transformation system relative to parent's tm.
		TimeInterval iv;
		const AffineTransformation& parentTM = node->parentNode()->getWorldTransform(time, iv);
		transformSystem = transformSystem * parentTM.inverse();

		// Rotate node in transformation system.
		Rotation scaledRot = Rotation(rot.axis, rot.angle * multiplier);
		node->transformationController()->rotate(time, scaledRot, transformSystem);

		// Translate node for off-center rotation.
		if(!ANIM_MANAGER.isAnimating()) {
			AffineTransformation inverseSys = transformSystem.inverse();
			/// Get node position in parent's space.
			AffineTransformation curTM;
			node->transformationController()->getValue(time, curTM, iv);
			Point3 nodePos = ORIGIN + curTM.getTranslation();
			nodePos = inverseSys * nodePos;
			Vector3 translation = (AffineTransformation::rotation(scaledRot) * nodePos) - nodePos;
			node->transformationController()->translate(time, translation, transformSystem);
		}
	}
}

};
