/***************************************************************************
 *   Copyright (C) 2008 by Mark Jaroski <mark@geekhive.net>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "plasma-wifi-signal.h"

#include <QPainter>
#include <QFontMetrics>
#include <QStyleOptionGraphicsItem>
#include <QSizeF>
#include <QRectF>
#include <QFile>
#include <KGlobalSettings>
 
#include <plasma/svg.h>
#include <plasma/theme.h>
 

Plasma_WiFi_Signal_Strength::Plasma_WiFi_Signal_Strength(
                    QObject *parent, const QVariantList &args  )
    : Plasma::Applet(parent, args),
      m_signal_strength(0)
{
    // this will get us the standard applet background, for free!
//    setDrawStandardBackground(true);
    resize(200, 200);
}
 
Plasma_WiFi_Signal_Strength::~Plasma_WiFi_Signal_Strength()
{
	m_timer->stop();
	delete m_timer;
}

void Plasma_WiFi_Signal_Strength::init()
{
    collectData();

	m_timer = new QTimer(this);
	connect(m_timer, SIGNAL(timeout()), this, SLOT(refresh()));
	m_timer->start(3000);

	m_boxAlpha = 128;
	m_boxColor.setAlpha(m_boxAlpha);
//    m_textColor = Plasma::Theme::self()->textColor();
//    m_boxColor = Plasma::Theme::self()->backgroundColor();

} 
 
void Plasma_WiFi_Signal_Strength::refresh()
{
    kDebug() << "refreshing";
    collectData();
    update();
}
 
void Plasma_WiFi_Signal_Strength::collectData()
{
	QFile wireless_stats("/proc/net/wireless", this);
	if ( !wireless_stats.open(QIODevice::ReadOnly | QIODevice::Text) ) {
		kDebug() << "File /proc/net/wireless could not be opened";
		return;
	}
	QString contents = wireless_stats.read(65535);
    QStringList lines = contents.split('\n');
    if(lines.size() > 2) {
        QStringList parts = lines[2].split( QRegExp( "\\s+" ) );
        if(parts.size() > 4) {
            kDebug() << " parts[3]:" << parts[3];
            m_signal_strength = parts[3].left(parts[3].size() - 1).toInt();
        }
    }
	wireless_stats.close();
}

void Plasma_WiFi_Signal_Strength::paintInterface(QPainter *p,
        const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED( option );

    p->setRenderHint(QPainter::SmoothPixmapTransform);
    p->setRenderHint(QPainter::Antialiasing);
 
    // We place the icon and text
    int width = (int)contentsRect.width();
    int height = (int)contentsRect.height();
    paintAntenna( p, contentsRect );
    p->save();
    p->setPen(Qt::white);
    if ( height > DRAW_LABEL_SIZE_LIMIT && width > DRAW_LABEL_SIZE_LIMIT ) {
        paintLabel(p, contentsRect, QString::number(m_signal_strength) + "%");
    }
    p->restore();
}

void Plasma_WiFi_Signal_Strength::paintAntenna( QPainter *p, const QRect &contentsRect )
{
    QString svg_file = QString();
    if ( (int)contentsRect.width() > 100 ) {
        svg_file = "widgets/network-wireless";
    } else if ( (int)contentsRect.width() > 32 ) {
        svg_file = "widgets/network-wireless-48x48";
    } else {
        svg_file = "widgets/network-wireless-16x16";
    }
    Plasma::Svg* svg;
    svg = new Plasma::Svg(this);
    svg->setImagePath( svg_file );
    svg->setContainsMultipleImages( false );
//    svg->resize(contentSize());

    if (svg->hasElement("Antenna")) 
            svg->paint(p, contentsRect, "Antenna");

    QString signal_layer = QString();
    if ( m_signal_strength == 0 ) {
        signal_layer = "off";
    } else if ( m_signal_strength < 10 ) {
        signal_layer = "on";
    } else if ( m_signal_strength < 20 ) {
        signal_layer = "on_1";
    } else if ( m_signal_strength < 30 ) {
        signal_layer = "on_2";
    } else if ( m_signal_strength < 40 ) {
        signal_layer = "on_3";
    } else if ( m_signal_strength < 50 ) {
        signal_layer = "on_4";
    } else if ( m_signal_strength < 60 ) {
        signal_layer = "on_5";
    } else if ( m_signal_strength < 70 ) {
        signal_layer = "on_6";
    } else if ( m_signal_strength < 70 ) {
        signal_layer = "on_8";
    } else {
        signal_layer = "on_9";
    }

    kDebug() << signal_layer;

    if (svg->hasElement( signal_layer )) 
            svg->paint( p, contentsRect, signal_layer );

}

// adapted from the battery plasmoid
void Plasma_WiFi_Signal_Strength::paintLabel(QPainter *p, 
            const QRect &contentsRect, const QString& labelText)
{
    // Store font size, we want to restore it shortly
    int original_font_size = m_font.pointSize();

    // Fonts smaller than smallestReadableFont don't make sense.
    m_font.setPointSize(qMax(KGlobalSettings::smallestReadableFont().pointSize(), m_font.pointSize()));
    QFontMetrics fm(m_font);
    int text_width = fm.width(labelText);

    // Longer texts get smaller fonts
    if (labelText.length() > 4) {
        if (original_font_size/1.5 < KGlobalSettings::smallestReadableFont().pointSize()) {
            m_font.setPointSize((int)(KGlobalSettings::smallestReadableFont().pointSize()));
        } else {
            m_font.setPointSize((int)(original_font_size/1.5));
        }
        fm = QFontMetrics(m_font);
        text_width = (int)(fm.width(labelText) * 1.2);
    } else {
        // Smaller texts get a wider box
        text_width = (int)(text_width * 1.4);
    }
    if (formFactor() == Plasma::Horizontal ||
            formFactor() == Plasma::Vertical) {
        m_font = KGlobalSettings::smallestReadableFont();
        m_font.setWeight(QFont::Bold);
        fm = QFontMetrics(m_font);
        text_width = (int)(fm.width(labelText)+8);
    } 
    p->setFont(m_font);

    // Let's find a good position for painting the background
    QRect text_rect = QRect(contentsRect.left()+(contentsRect.width()-text_width)/2,
                            (int)(contentsRect.top()+( (contentsRect.height() - (int)fm.height()) * 0.85 )),
                            text_width, (int)(fm.height()*1.2));

    p->setBrush(m_boxColor);

    // Find sensible proportions for the rounded corners
    float round_prop = text_rect.width() / text_rect.height();

    // Tweak the rounding edge a bit with the proportions of the textbox
    int round_radius = 35;
    p->drawRoundRect(text_rect, (int)(round_radius/round_prop), round_radius);

    p->setBrush(m_textColor);
    p->drawText(text_rect, Qt::AlignCenter, labelText);

    // Reset font and box
    m_font.setPointSize(original_font_size);
    m_boxColor.setAlpha(m_boxAlpha);
}

Qt::Orientations Plasma_WiFi_Signal_Strength::expandingDirections() const
{
    // no use of additional space in any direction
    return 0;
}

 
#include "plasma-wifi-signal.moc" 
