/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

struct spline_params {
    double k, m;
};

struct spline {
    double (*points)[2];
    int n;
    struct spline_params params[0];
};

int spline_sort_comp(const void *a, const void *b)
{
    double x1 = *(double *)a;
    double x2 = *(double *)b;
    return (int)(x1 - x2);
}

/*
 * Initializes a monotone cubic Hermite spline for interpolating
 * between the input points, using the Fritsch–Carlson method.
 *
 *   http://en.wikipedia.org/wiki/Monotone_cubic_interpolation
 *
 * The interpolated points are guaranteed to have piecewise
 * monotonicity, and will be fully monotone if the input points
 * are monotone.
 */
struct spline *spline_new(double (*points)[2], int num_points)
{
    struct spline *s;
    struct spline_params *param;
    double (*p)[2];
    int i;
    double alpha, beta, prev_beta;

    s = malloc(sizeof(struct spline) +
               sizeof(struct spline_params) * num_points);
    if (!s)
        return NULL;
    s->points = malloc(2 * sizeof(double) * num_points);
    if (!s->points) {
        free(s);
        return NULL;
    }
    s->n = num_points;

    /*
     * Input points are likely to already be in order of increasing
     * x, but sort them just in case.
     */
    memcpy(s->points, points, 2 * sizeof(double) * num_points);
    qsort(s->points, num_points, 2 * sizeof(double), spline_sort_comp);

    p = s->points;
    param = s->params;

    for (i = 0; i < s->n - 1; i++)
        param[i].k = (p[i+1][1] - p[i][1]) / (p[i+1][0] - p[i][0]);

    param[0].m = param[0].k;
    for (i = 1; i < s->n; i++)
        param[i].m = (param[i-1].k + param[i].k) / 2;
    param[s->n - 1].m = param[s->n - 2].k;

    beta = 0;
    for (i = 0; i < s->n - 1; i++) {
        if (p[i][1] == p[i+1][1]) {
            param[i].m = 0;
            param[++i].m = 0;
            beta = 0;
            continue;
        }
        prev_beta = beta;
        alpha = param[i].m / param[i].k;
        beta = param[i+1].m / param[i].k;
        if (alpha < 0 || prev_beta < 0) {
            param[i].m = 0;
            continue;
        }
        if (alpha > 3)
            param[i].m = 3 * param[i].k;
        if (beta > 3)
            param[i+1].m = 3 * param[i].k;
    }

    return s;
}

void spline_free(struct spline *s)
{
    free(s->points);
    free(s);
}

double spline_interpolate(struct spline *s, double x)
{
    int i;
    double h, t, ret;

    for (i = 0; i < s->n - 1; i++) {
        if (x >= s->points[i][0] && x < s->points[i+1][0])
            break;
    }

    /* Handle out-of-bounds and boundary cases */
    if (i == 0 && x <= s->points[0][0])
        return s->points[0][1];
    if (i >= s->n - 1)
        return s->points[s->n - 1][1];

    h = s->points[i+1][0] - s->points[i][0];
    t = (x - s->points[i][0]) / h;
    ret = s->points[i][1] * (1 + t * t * (2 * t - 3));
    ret += h * s->params[i].m * t * (1 + t * (t - 2));
    ret += s->points[i+1][1] * t * t * (3 - 2 * t);
    ret += h * s->params[i+1].m * t * t * (t - 1);
    return ret;
}
