/*
 * Copyright(c) 1997 by Jun-ichiro Itoh. All rights reserved.
 * Freely redistributable.  Absolutely no warranty.
 *
 * author contact: Jun-ichiro itojun Itoh <itojun@mt.cs.keio.ac.jp>
 * $Id: main.c,v 1.2 1998/08/12 14:31:59 itojun Exp $
 */
#include "common.h"

#if HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#if HAVE_TERMIOS_H
# include <termios.h>
#else
/*# if HAVE_TERMIO_H*/
# if 0
#  include <termio.h>
# else
#  include <sgtty.h>
# endif
#endif

/* screen */
static int echo_x;
static int echo_y;
static int echo_maxy;

/* serial lines */
static int baud = 9600;
static char *n0, *n1;
static int fd0, fd1;
static u_int32_t incnt0, incnt1;

/* console */
static int consconnect;

/* uid */
static uid_t uid, euid;
static gid_t gid, egid;
static int uidswapped = 0;

/* uucp locking */
static int useuucplock = 1;
extern int uu_lock __P((char *));
extern int uu_unlock __P((char *));

void
daemonuid()
{
	if (uidswapped == 0)
		return;

#ifdef HAVE_SETREUID
	setreuid(uid, euid);
	setregid(gid, egid);
#else
	setuid(uid);
	seteuid(euid);
	setgid(gid);
	setegid(egid);
#endif
	uidswapped = 0;
}

void
useruid()
{
	if (uidswapped == 1)
		return;

#ifdef HAVE_SETREUID
	setregid(egid, gid);
	setreuid(euid, uid);
#else
	setgid(egid);
	setegid(gid);
	setuid(euid);
	seteuid(uid);
#endif
	uidswapped = 1;
}

static int
uucplock(name)
	char *name;
{
	int ret;

	if (useuucplock) {
		daemonuid();
		ret = uu_lock(rindex(name, '/') + 1);
		useruid();
	} else
		ret = 0;
	return ret;
}

static int
uucpunlock(name)
	char *name;
{
	int ret;

	if (useuucplock) {
		daemonuid();
		ret = uu_unlock(rindex(name, '/') + 1);
		useruid();
	} else
		ret = 0;
	return ret;
}

static char *
canondev(name)
	char *name;
{
	int fd;
	char *p;

	daemonuid();

	p = malloc(strlen(name) + 1);
	if (!p) {
		perror("malloc");
		exit(1);
	}
	strcpy(p, name);
	fd = open(p, O_RDWR|O_NONBLOCK);
	if (0 <= fd) {
		close(fd);
		goto back;
	}

	p = realloc(p, strlen(name) + 6);
	if (!p) {
		perror("realloc");
		exit(1);
	}
	strcpy(p, "/dev/");
	strcat(p, name);
	fd = open(p, O_RDWR|O_NONBLOCK);
	if (0 <= fd) {
		close(fd);
		goto back;
	}

	free(p);
	p = NULL;
back:
	useruid();
	return p;
}

void setbaud(fd, baud)
	int fd;
	int baud;
{
#if HAVE_TERMIOS_H
	/* termios */
	struct termios tio;

	if (tcgetattr(fd, &tio) < 0) {
		perror("tcgetattr");
		uucpunlock(n0);
		uucpunlock(n1);
		exit(1);
	}
	tio.c_iflag = 0;
	tio.c_oflag = 0;
	tio.c_cflag = CS8 | CREAD | CLOCAL;
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 5;
	cfsetispeed(&tio, baud);
	cfsetospeed(&tio, baud);
	if (tcsetattr(fd, TCSANOW, &tio) < 0) {
		perror("tcsetattr");
		uucpunlock(n0);
		uucpunlock(n1);
		exit(1);
	}
#else
/*#if HAVE_TERMIO_H*/
# if 0
	/* termio */
# else
	/* sgtty */
	struct sgttyb ttyb;

	if (ioctl(fd, TIOCGETP, &ttyb) < 0) {
		perror("ioctl(TIOCGETP)");
		uucpunlock(n0);
		uucpunlock(n1);
		exit(1);
	}
	ttyb.sg_ispeed = baud;
	ttyb.sg_ospeed = baud;
	ttyb.sg_flags = 0;
	if (ioctl(fd, TIOCSETP, &ttyb) < 0) {
		perror("ioctl(TIOCSETP)");
		uucpunlock(n0);
		uucpunlock(n1);
		exit(1);
	}
# endif
#endif
}

void
showstatus(line, state, speed, name, cnt)
	int line;
	int state;
	int speed;
	char *name;
	u_int32_t cnt;
{
#define	SHOW(str, bit)	\
    {					\
	if (state & bit)		\
		standout();		\
	addstr(str);			\
	standend();			\
	addch(' ');			\
    }

	move(line, 0);
	clrtoeol();
	SHOW("LE", TIOCM_LE);
	SHOW("DTR", TIOCM_DTR);
	SHOW("RTS", TIOCM_RTS);
	SHOW("ST", TIOCM_ST);
	SHOW("SR", TIOCM_SR);
	SHOW("CTS", TIOCM_CTS);
	SHOW("CD", TIOCM_CD);
	SHOW("RI", TIOCM_RI);
	SHOW("DSR", TIOCM_DSR);

	printw(" %dbps  %s  %lu chars ", speed, name, cnt);

#undef	SHOW()
}

void
echoback_init()
{
	size_t i;

	echo_x = 0;
	echo_y = 3;
	echo_maxy = echo_y + ((scrn_rows - echo_y) / 3) * 3 - 1;

	move(2, 0);
	for (i = 0; i < scrn_cols; i++)
		addch('-');
}

void
echoback(line, buf, len)
	int line;
	char *buf;
	int len;
{
	size_t i;

#define CURSOR_SYNC					\
    {							\
	if (scrn_cols <= echo_x) {			\
		echo_x -= scrn_cols;			\
		echo_y += 3;				\
	}						\
	if (echo_maxy < echo_y) {			\
		move(3, 0);				\
		deleteln();				\
		deleteln();				\
		deleteln();				\
		move(scrn_rows - 4, 0);			\
		insertln();				\
		insertln();				\
		insertln();				\
		echo_y -= 3;				\
	}						\
	move(echo_y + line, echo_x);			\
    }


	move(echo_y + line, echo_x);
	i = 0;
	while (i < len) {
		CURSOR_SYNC;

		if (buf[i] == ' ') {
			standout();
			addch(' ');
			standend();
			echo_x++;
			i++;
		} else if (buf[i] != ' ' && isprint(buf[i])) {
			addch(buf[i]);
			echo_x++;
			i++;
		} else {
			if (echo_x + 1 < scrn_cols) {
				standout();
				printw("%02x", buf[i] & 0xff);
				standend();
				echo_x += 2;
				i++;
			} else {
				/* scroll this line */
				echo_x++;
			}
		} 
	}

	CURSOR_SYNC;
#undef CURSOR_SYNC
}

char *
msg(prompt)
	char *prompt;
{
	move(scrn_rows - 1, 0);
	clrtoeol();
	addstr(prompt);
}

char *
ask(prompt)
	char *prompt;
{
	int c;
	static char buf[80];
	char *ret;
	int idx;
	int max;
	int y, x;

	move(scrn_rows - 1, 0);
	clrtoeol();
	addstr(prompt);

	idx = 0;
	max = scrn_cols - strlen(prompt) - 2;
	while (1) {
		refresh();
		switch (c = getch()) {
		case '\n':
		case '\r':
			goto gotit;
		case '\033':
			ret = NULL;
			goto quit;
		case '\b':
			if (!idx) {
				scrn_beep();
				break;
			}
			getyx(stdscr, y, x);
			move(y, x - 1);
			addch(' ');
			move(y, x - 1);
			buf[--idx] = '\0';
			break;
		case 'U' & 0x1f:
			if (!idx) {
				scrn_beep();
				break;
			}
			getyx(stdscr, y, x);
			move(y, x - idx);
			clrtoeol();
			buf[idx = 0] = '\0';
			break;
		default:
			if (c < ' ' || max <= idx) {
				scrn_beep();
				break;
			}
			addch(c);
			buf[idx++] = c;
			break;
		}
	}
gotit:
	buf[idx] = '\0';
	ret = buf;
quit:
	move(scrn_rows - 1, 0);
	clrtoeol();
	return ret;
}

int
cmd()
{
	int c;
	char *tmp;
	int line;

	switch (c = getch()) {
	case 'Q':
		return 0;	/*let me quit*/
	case 'b':
		tmp = ask("new baudrate? ");
		if (tmp) {
			baud = atoi(tmp);
			setbaud(fd0, baud);
			setbaud(fd1, baud);
		}
		break;
	case 'c':
		incnt0 = incnt1 = 0;
		break;
	case 'C':
		tmp = ask("line to connect console to? ");
		if (!tmp)
			break;
		consconnect = atoi(tmp);
		if (consconnect == 0) {
			msg("console is acting as line 0. "
				"special keys: ^V/^X/ESC");
		} else if (consconnect == 1) {
			msg("console is acting as line 1. "
				"special keys: ^V/^X/ESC");
		} else {
			msg("no such line. line must be 0 or 1.");
			consconnect = -1;
		}
		break;
	case 'B':
		tmp = ask("which line you would like to make a binary log? ");
		if (!tmp || strlen(tmp) == 0)
			break;
		line = atoi(tmp);
		if (line != 0 && line != 1) {
			msg("no such line. line must be 0 (logs 0>1) or 1 (logs 1>0).");
			break;
		}
		tmp = ask("binary logfile name? ");
		if (!tmp)
			break;
		binlog_end(line);
		if (strlen(tmp) == 0) {
			msg("binary logfile closed.");
			break;
		}
		if (binlog_init(line, tmp) < 0) {
			msg("error opening binary logfile.");
			break;
		}
		if (line == 0)
			msg("making binary log of datastream 0>1.");
		else
			msg("making binary log of datastream 1>0.");
		break;
	case 'L':
		tmp = ask("textual logfile name? ");
		if (!tmp)
			break;
		log_end();
		if (strlen(tmp) == 0) {
			msg("textual log file closed.");
			break;
		}
		if (log_init(tmp) < 0) {
			msg("error opening textual logfile.");
			break;
		}
		msg("making textual log.");
		break;
	case 'L' & 0x1f:
		clear();
		echoback_init();
		break;
	case 'm':
		tmp = ask("memo? ");
		if (!tmp)
			break;
		log_memo(tmp);
		break;
	}

	return 1;	/*continue execution*/
}

static int
swapctrl(m)
	int m;
{
	int n;

	n = m;
	/* think about cross cable. */
#if 0
	n &= ~TIOCM_CTS;
	if (m & TIOCM_RTS)
		n |= TIOCM_CTS;
#endif
	n &= ~TIOCM_RTS;
	if (m & TIOCM_CTS)
		n |= TIOCM_RTS;
	n &= ~TIOCM_DTR;
	if (m & (TIOCM_DSR | TIOCM_CAR))
		n |= TIOCM_DTR;
#if 0
	n &= ~TIOCM_DSR | TIOCM_CAR;
	if (m & TIOCM_DTR)
		n |= (TIOCM_DSR | TIOCM_CD);
#endif
	return n;
}

void
mainloop()
{
	fd_set rdfd;
	int maxfd;
	struct timeval t;
	char buf[1024];
	int len;
	int m0;
	int m1;
	int om0;
	int om1;
	int nm0;
	int nm1;
	int c;
	size_t i;
	char *tmp;

	maxfd = (fd0 > fd1) ? fd0 : fd1;

	incnt0 = incnt1 = 0;
	consconnect = -1;
	om0 = om1 = -1;

	msg(VERSION);
	while (1) {
		/* we always do this. */
		if (ioctl(fd0, TIOCMGET, &m0) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMGET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}
		if (ioctl(fd1, TIOCMGET, &m1) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMGET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}

		nm0 = swapctrl(m1);
		nm1 = swapctrl(m0);

		if (ioctl(fd0, TIOCMSET, &nm0) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMSET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}
		if (ioctl(fd1, TIOCMSET, &nm1) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMSET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}

		if (ioctl(fd0, TIOCMGET, &m0) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMGET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}
		if (ioctl(fd1, TIOCMGET, &m1) < 0) {
			scrn_end();
			log_end();
			binlog_end(0);
			binlog_end(1);
			perror("ioctl(TIOCMGET)");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		}

		showstatus(0, m0, baud, n0, incnt0);
		showstatus(1, m1, baud, n1, incnt1);
		if (om0 != m0)
			log_status(0, m0, n0);
		if (om1 != m1)
			log_status(1, m1, n1);
		om0 = m0;
		om1 = m1;
		move(echo_y, echo_x);
		refresh();

		/* command/chars */
		FD_ZERO(&rdfd);
		FD_SET(fd0, &rdfd);
		FD_SET(fd1, &rdfd);
		FD_SET(STDIN_FILENO, &rdfd);
		/* interval: 1sec */
		t.tv_sec = 1;
		t.tv_usec = 0;
		switch (select(maxfd + 1, &rdfd, NULL, NULL, &t)) {
		case -1:
			if (errno == EINTR)
				break;
			perror("select");
			uucpunlock(n0);
			uucpunlock(n1);
			exit(1);
		case 0:
			log_flush();
			binlog_flush(0);
			binlog_flush(1);
			continue;
		default:
			break;
		}

		if (FD_ISSET(fd0, &rdfd)) {
			len = read(fd0, buf, sizeof(buf));
			incnt0 += len;
			echoback(0, buf, len);
			log_write('0', '1', buf, len);
			binlog_write(0, buf, len);
			write(fd1, buf, len);
		}
		if (FD_ISSET(fd1, &rdfd)) {
			len = read(fd1, buf, sizeof(buf));
			incnt1 += len;
			echoback(1, buf, len);
			log_write('1', '0', buf, len);
			binlog_write(1, buf, len);
			write(fd0, buf, len);
		}
		if (FD_ISSET(STDIN_FILENO, &rdfd)) {
			if (consconnect == 0 || consconnect == 1) {
				c = getch();
				c &= 0xff;
				if (c == '\033') {
					consconnect = -1;
					msg("console is disconnected "
						"from line.");
					continue;
				} else if (c == ('V' & 0x1f)) {
					c = getch();
					goto gotch;
				} else if (c == ('X' & 0x1f)) {
					int c1, c2;
					char *hex = "0123456789ABCDEF";

					c1 = getch();
					if (islower(c1))
						c1 = toupper(c1);
					c2 = getch();
					if (islower(c2))
						c2 = toupper(c2);
					if (!index(hex, c1)) {
						scrn_beep();
						continue;
					}
					if (!index(hex, c2)) {
						scrn_beep();
						continue;
					}
					c = (index(hex, c1) - hex) & 0x0f;
					c = c << 4;
					c |= (index(hex, c2) - hex) & 0x0f;
					goto gotch;
				}
gotch:
				buf[0] = c;
				echoback(consconnect, buf, 1);
				if (consconnect == 0)
					log_write('C', '1', buf, 1);
				else
					log_write('C', '0', buf, 1);
				write(consconnect == 0 ? fd1 : fd0, buf, 1);
				continue;
			}

			if (cmd() == 0)
				goto end;
		}
	}

end:
	log_end();
	binlog_end(0);
	binlog_end(1);
}

void
main(argc, argv)
	int argc;
	char **argv;
{
	char *s;

	uid = getuid();
	euid = geteuid();
	gid = getgid();
	egid = getegid();
	useruid();

	useuucplock = 1;

	while (--argc > 0 && **++argv == '-') {
		for (s = argv[0] + 1; *s != '\0'; s++) {
			switch (*s) {
			case 'b':
				baud = atoi(s + 1);
				while (*s)
					s++;
				s--;
				break;
			case 'u':
				useuucplock = 0;
				break;
			default:
				fprintf(stderr, "unknown option -%c\n", *s);
				exit(1);
			}
		}
	}

	if (argc != 2) {
		fprintf(stderr, "no device specified\n");
		exit(1);
	}
	n0 = canondev(argv[0]);
	n1 = canondev(argv[1]);
	if (!n0 || !n1) {
		fprintf(stderr, "inappropriate devices specified\n");
		exit(1);
	}

	if (uucplock(n0) < 0) {
		fprintf(stderr, "device %s in use specified\n", n0);
		exit(1);
	}
	if (uucplock(n1) < 0) {
		uucpunlock(n0);
		fprintf(stderr, "device %s in use specified\n", n0);
		exit(1);
	}

	daemonuid();

	fd0 = open(n0, O_RDWR|O_NDELAY);
	if (fd0 < 0) {
		perror("open");
		exit(1);
	}
	fd1 = open(n1, O_RDWR|O_NDELAY);
	if (fd1 < 0) {
		perror("open");
		exit(1);
	}

	setbaud(fd0, baud);
	setbaud(fd1, baud);

	useruid();

	scrn_init();
	echoback_init();

	mainloop();

	close(fd0);
	close(fd1);
	uucpunlock(n0);
	uucpunlock(n1);
	scrn_end();
}
