-- lib/tongue/langpack.lua
--
-- Lua I18N library 'Tongue' - Language Packs
--
-- Copyright 2016 Daniel Silverstone <dsilvers@digital-scurf.org>
--
-- For licence terms, see COPYING
--

--- Language packs comprise zero or more sources of translations for a given
-- "language".
--
-- Packs may be added to at any time.  A pack can have a parent pack which will
-- be used if the pack in question lacks a translation token.  Tongue defines a
-- single default pack (available as tongue.pack.fallback) which always returns
-- a string comprising the token and its arguments flattened as strings.  This
-- is the only part of Tongue which assumes an ability to flatten token
-- arguments as strings, all other parts of tongue preserve token arguments
-- unchanged which allows the passing of complex arguments such as repository
-- objects.
--
-- @module tongue.langpack

local fallback, langpack = nil, {}

--- Fallback language pack.
--
-- The fallback language pack provides a mechanism to ensure that language
-- packs can always expand a translation to some extent.
--
-- @field fallback

--- Try and convert a string to a function.
--
-- This takes a tongue expansion string and tries to convert it to a function
-- which can be used to generate the expansion later.
--
-- @tparam string expn The expansion string
-- @treturn[1] function The resulting function
-- @treturn[2] nil To indicate error
-- @treturn[2] string Error message
local function try_convert(expn)
   local expn_t = {}
   local pstate = "normal"
   local acc = ""
   local function transacc(repl)
      if acc ~= "" then
	 expn_t[#expn_t+1] = acc
      end
      acc = repl or ""
   end
   local parser = {
      normal = function(ch)
	 if ch == "$" then
	    pstate = "dollar"
	 else
	    acc = acc .. ch
	 end
      end,
      dollar = function(ch)
	 if ch == "{" then
	    pstate = "getname"
	    transacc({type="arg", name=""})
	 elseif ch == "(" then
	    pstate = "getexpn"
	    transacc({type="expn", name=""})
	 else
	    acc = acc .. "$" .. ch
	    pstate = "normal"
	 end
      end,
      getname = function(ch)
	 if ch == "}" then
	    transacc()
	    pstate = "normal"
	 else
	    acc.name = acc.name .. ch
	 end
      end,
      getexpn = function(ch)
	 if ch == ")" then
	    transacc()
	    pstate = "normal"
	 elseif ch == "," then
	    pstate = "getarg"
	    acc[#acc+1] = ""
	 else
	    acc.name = acc.name .. ch
	 end
      end,
      getarg = function(ch)
	 if ch == ")" then
	    transacc()
	    pstate = "normal"
	 elseif ch == "=" then
	    pstate = "getvalue"
	    acc[#acc] = {k=acc[#acc],v=""}
	 elseif ch == "," then
	    acc[#acc+1] = ""
	 else
	    acc[#acc] = acc[#acc] .. ch
	 end
      end,
      getvalue = function(ch)
	 if ch == ")" then
	    transacc()
	    pstate = "normal"
	 elseif ch == "," then
	    pstate = "getarg"
	    acc[#acc+1] = ""
	 else
	    acc[#acc].v = acc[#acc].v .. ch
	 end
      end,
   }
   for ch in expn:gmatch("(.)") do
      local fn = parser[pstate]
      if fn == nil then
	 return nil, "Unexpected parse state: " .. tostring(pstate)
      end
      fn(ch)
   end
   if pstate ~= "normal" then
      return nil, "Unexpected end of string parsing expansion expression"
   end
   transacc()
   -- Now we need to validate the expansion table
   -- Simple rules on expansion expressions:
   -- 1. must have a name
   -- 2. every argument must have a name (str) or a key/value (tab)
   -- 3. maximum of one ... as a name, not a key/value
   -- 4. any key/value must have the key and value be non-empty
   for _, t in ipairs(expn_t) do
      if type(t) == "string" then
	 -- Nothing to check here
      else
	 if t.name == "" then
	    return nil, "Empty name in expansion expression"
	 end
	 local found_dotdotdot = false
	 if t.type == "expn" then
	    for _, tt in ipairs(t) do
	       if tt == "..." then
		  if found_dotdotdot then
		     return nil, "Repeated ... in expansion expression"
		  end
		  found_dotdotdot = true
	       elseif type(tt) == "table" then
		  if tt.k == "" or tt.v == "" then
		     return nil, "Empty key or value in expansion expression"
		  end
		  if tt.k == "..." then
		     return nil, "Use of ... in key=value expansion expression"
		  end
	       end
	    end
	 else
	    -- Nothing to check here, arg expansions are simple
	 end
      end
   end
   -- Okay, so superficially things look good, wrapper it up in a function
   local function processed_expansion(args)
      -- Note: this closes over expn_t from the outer function
      local ret = {}
      for _, t in ipairs(expn_t) do
	 if type(t) == "string" then
	    ret[#ret+1] = t
	 elseif t.type == "expn" then
	    -- We're "recursing"...
	    local token = t.name
	    local nargs = {}
	    for _, tt in ipairs(t) do
	       if type(tt) == "string" then
		  if tt == "..." then
		     for k, v in pairs(args) do
			nargs[k] = v
		     end
		  else
		     nargs[tt] = args[tt]
		  end
	       else
		  nargs[tt.k] = tt.v
	       end
	    end
	    nargs._pack = args._pack
	    ret[#ret+1] = args._pack:expand(token, nargs)
	 else
	    -- Argument expansion
	    ret[#ret+1] = args[t.name] or ""
	 end
      end
      return table.concat(ret, "")
   end
   return processed_expansion
end

--- Generate a function to localise when a conversion failed.
--
-- This generates a function to report a failed expansion and why
--
-- @tparam string token The token which failed to expand.
-- @tparam string expn The content of that token.
-- @tparam string msg The error message.
-- @treturn function a function which can report that failure
local function report_conversion(token, expn, msg)
   return function(args)
      local nargs = {
	 _pack = args.pack,
	 token = token,
	 expn = expn,
	 msg = msg,
	 args = args
      }
      return args._pack:expand("__TONGUE.FAILEDEXPAND", nargs)
   end
end


--- Tongue Language Pack.
--
-- A Tongue language pack comprises zero or more translations associated with
-- the language (and sub-language) chosen at construction time.  Language packs
-- may have a parent and can be augmented at any time with further
-- translations.
--
-- @type langpack

--- Add a token expansion to a pack.
--
-- This adds the expansion of a token to a tongue language pack.  On addition,
-- if the expansion is a string then it will *NOT* be validated unless the
-- strict argument is set.
--
-- Token names are automatically uppercased in the ASCII charset.
--
-- @tparam string token The token to be expanded
-- @tparam string|function expansion The expansion string (or function)
-- @tparam bool strict Whether to treat a bad expansion string as an error.
-- @treturn bool Whether or not the expansion was successfully added
-- @function add_token

function langpack:add_token(token, expansion, strict)
   if strict and type(expansion) == "string" then
      expansion = try_convert(expansion)
      if not expansion then
	 -- We don't give a reason, we just fail to expand/add
	 return false
      end
   end
   self.entries[token:upper()] = expansion
   return true
end

--- Expand a token and arguments into a message.
--
-- This expands the given token and arguments into a full message.  This will
-- always succeed unless something has gone crazywrong with the internals of
-- tongue or the language packs.
--
-- Passed-in tokens are always uppercased before expansion.
--
-- @tparam string token The token to be expanded
-- @tparam table args Arguments to the token expansion
-- @treturn string The expanded result
-- @function expand
function langpack:expand(token, args)
   local pack = self
   args = args or {}
   if not args._pack then args._pack = pack end
   local ok, expn = pcall(function()
	 if not pack.entries[token] then
	    -- expand upward toward the parent
	    return pack.parent:expand(token, args)
	 end
	 if type(pack.entries[token]) == "string" then
	    -- Attempt to convert the string form into a function
	    local fn, msg = try_convert(pack.entries[token])
	    if not fn then
	       -- Failed to convert, so generate an error report expn. instead
	       fn = report_conversion(token, pack.entries[token], msg)
	    end
	    pack.entries[token] = fn
	 end
	 return pack.entries[token](args)
   end)
   if ok then
      return expn
   end
   -- Failed to expand for whatever reason, we need to report this:
   return self:expand("__TONGUE.INTERNALERROR",
		      { token=token, args=args, err=expn })
end


-- set up the language pack metatable
local langpack_mt = {
   __index = langpack
}

---
-- @section tongue.langpack

--- Create a language pack.
--
-- This creates a tongue language pack.  Once the pack is created it can be
-- populated in various ways by language loaders.  The passed in language
-- is used by other parts of Tongue (e.g. the message resolver) to manage
-- language packs.
--
-- @tparam string language The language name (e.g. 'en')
-- @tparam ?string sublang  The sub-language name (or nil if unwanted) e.g. "GB"
-- @tparam ?langpack parent The parent langauge pack (if nil, Tongue will use the fallback)
-- @treturn langpack The newly created language pack
-- @function create
local function createpack(language, sublang, parent)
   local retpack = {}
   retpack.language = language
   retpack.sublang = sublang
   retpack.parent = parent or fallback
   retpack.entries = {}
   retpack.lang = sublang and language .. "_" .. sublang or language
   return setmetatable(retpack, langpack_mt)
end

-- Finally, populate the fallback pack with some cleverness
fallback = createpack()

local function flatten(t)
   if t._VISITED then return "???" end
   t._VISITED=t
   local ret = {}
   for k, v in pairs(t) do
      if type(k) == "string" and string.sub(k, 1, 1) ~= "_" then
	 if type(v) == "table" then
	    v = flatten(v)
	 elseif type(v) == "string" then
	    v = ("%q"):format(v)
	 else
	    v = tostring(v)
	 end
	 ret[#ret+1] = k .. "=" .. v
      end
   end
   t._VISITED=nil
   return "{" .. table.concat(ret,",") .. "}"
end

local function fallback_entries_index(t, name)
   local f = function(args)
      return "!!!" .. name .. "!!!" .. flatten(args)
   end
   t[name] = f
   return f
end

setmetatable(fallback.entries, {__index=fallback_entries_index})

fallback.entries["__TONGUE.INTERNALERROR"] = function(args)
   -- args.token
   -- args.err
   -- args.args
   return table.concat({
	 "!!!Internal error while processing", tostring(args.token),
	 "-", tostring(args.err), "-", flatten(args.args)}, " ")
end

fallback.entries["__TONGUE.FAILEDEXPAND"] = function(args)
   -- args.token
   -- args.expn
   -- args.msg
   -- args.args
   return table.concat({
	 "!!!Failed to expand", tostring(args.token), "-", tostring(args.expn),
	 "-", tostring(args.msg), "-", flatten(args.args)}, " ")
end

return {
   create = createpack,
   fallback = fallback
}
