# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from elisa.extern.translation import Translator

from elisa.core import log, common

from elisa.core import config, plugin_registry
from elisa.core import interface_controller
from elisa.core import metadata_manager
from elisa.core import input_manager
from elisa.core import bus
from elisa.extern.log import log as extern_log

from twisted.trial import unittest
import os, sys, inspect



DEFAULT_CONFIG = """\
[general]
version = '%(version)s'
install_date = '%(install_date)s'
metadata_providers = []
service_providers = []

[media_db:media_scanner]
enabled = '1'
fivemin_location_updates = []
hourly_location_updates = []
daily_location_updates = []
weekly_location_updates = []
unmonitored_locations = []

[media_db:db]
db_backend = 'sqlite'
database = 'elisa.db'

[xmlmenu:locations_builder]
locations = []
auto_locations = 1

[lirc:lirc_input]
# filename of the LIRC config map to use
lirc_rc = 'streamzap.lirc'
delay = '4'
repeat = '1'

[coherence:coherence_service]
logmode = 'none'
controlpoint = 'yes'

[[plugins]]

"""


class BoilerPlateApp:

    def __init__(self, config_file, default_config=None, load_all_plugins=False):
        #self.config = config.Config(config_file)
        if not default_config:
            default_config = DEFAULT_CONFIG
        self.errors = []
        
        default_config = default_config % {'version': 'test',
                                           'install_date': 'none'}
        self.config = config.Config('empty.conf', default_config)
        common.set_application(self)

        self.translator = Translator()
        manager = plugin_registry.PluginRegistry(self.config)
        if load_all_plugins:
            manager.load_plugins()
        
        self.plugin_registry = manager
        self.bus = bus.Bus()
        self.metadata_manager = metadata_manager.MetadataManager()
        self.input_manager = input_manager.InputManager()
        self.interface_controller = interface_controller.InterfaceController()
        self.interface_controller.initialize()
        
    def log_traceback(self):
        info = sys.exc_info()
        self.errors.append(info)

    def log_failure(self, failure):
        self.errors.append(failure)

class ElisaTestCase(unittest.TestCase, extern_log.Loggable):
    default_config = DEFAULT_CONFIG
    load_all_plugins = False
    tests_dir = os.path.dirname(__file__)
    
    def __init__(self, methodName='runTest'):
        self._booted = False
        
        test_file_path = inspect.getsourcefile(self.__class__)
        if test_file_path is not None:
            fname = os.path.basename(test_file_path)

            fname, _ = os.path.splitext(fname)
            self.logCategory = fname
            self.directory = os.path.dirname(test_file_path)
        
        #extern_log.init('ELISA_TESTS')
        unittest.TestCase.__init__(self, methodName=methodName)

    debug = extern_log.Loggable.debug
    info = extern_log.Loggable.info

    def _boot(self):
        if self._booted == False:
            if 'ELISA_DEBUG' not in os.environ:
                log.setDebug('*:0')

            app = BoilerPlateApp('empty.conf', self.default_config,
                                 self.load_all_plugins)
            self.debug("Set common.application to %r", app)
            
            self._booted = True
    
    def setUp(self):
        self._boot()
        
    def check_called(self, callback):
        self.failUnless(hasattr(callback, 'called'))
        self.assertEquals(callback.called, True)
