# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2007-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

"""
Functional tests for themoviedb.org resource provider.
"""

from twisted.trial.unittest import TestCase
from elisa.core.utils.misc import run_functional_tests_check
from twisted.internet import defer

from elisa.plugins.themoviedb.resource_provider import TheMovieDBResourceProvider
from elisa.plugins.themoviedb.constants import API_SERVER, IMG_SERVER
from elisa.plugins.themoviedb.utils import get_movie_lookup_url
from elisa.plugins.themoviedb.models import MovieModel, MoviesListModel
from elisa.plugins.base.models.media import RawDataModel

from elisa.core.media_uri import MediaUri

class TestTheMovieDBResourceProvider(TestCase):

    """
    This test case tests the TheMovieDBResourceProvider class.
    """

    def setUp(self):
        run_functional_tests_check()
        def created(provider):
            self.provider = provider
            return provider.initialize()

        create_dfr = TheMovieDBResourceProvider.create({}).addCallback(created)
        return create_dfr

    def tearDown(self):
        return self.provider.clean()

    def test_get_wrong_uri(self):
        """
        Test trying to get a unexisting URI.
        """
        uri = 'http://%s/idontexist' % API_SERVER
        result_model, request_defer = self.provider.get(MediaUri(uri))
        self.failUnlessFailure(request_defer, Exception)
        return request_defer

    def test_get_wrong_api_method(self):
        """
        Test trying to get an unknown method of the API
        """
        uri = 'http://%s/2.0/idontexist' % API_SERVER
        result_model, request_defer = self.provider.get(MediaUri(uri))
        self.failUnlessFailure(request_defer, NotImplementedError)
        return request_defer

    def _search_terminator(self):
        # I love that movie. so much.
        uri = get_movie_lookup_url('terminator')
        result_model, request_defer = self.provider.get(MediaUri(uri))
        return request_defer

    def test_search(self):
        """
        Test a simple API call.
        """
        def request_done(result_model):
            self.failUnless(isinstance(result_model, MoviesListModel))
            self.failUnless(reduce(lambda x, y: x and isinstance(y, MovieModel),
                                   result_model.movies))
            return result_model

        dfr = self._search_terminator()
        dfr.addCallback(request_done)
        return dfr

    def test_get_info(self):
        """
        Test a getInfo API call.
        """
        def got_info(info):
            self.failUnless(info.runtime is not None)

        def request_done(result_model):
            first_match = result_model.movies[0]
            self.failIf(first_match.runtime is not None)
            model, dfr = self.provider.get(first_match.api_url, first_match)
            dfr.addCallback(got_info)
            return dfr

        dfr = self._search_terminator()
        dfr.addCallback(request_done)
        return dfr

    def test_imdb_lookup(self):
        """
        Test a imdbLookup API call.
        """
        def got_info(info):
            self.failUnless(info.short_overview is not None)

        def request_done(result_model):
            first_match = result_model.movies[0]
            model, dfr = self.provider.get(first_match.imdb_url)
            dfr.addCallback(got_info)
            return dfr

        dfr = self._search_terminator()
        dfr.addCallback(request_done)
        return dfr

    def test_picture_retrieval(self):
        """
        Test the retrieval of a movie poster
        """
        def got_poster(poster):
            self.failUnless(isinstance(poster, RawDataModel))
            self.failUnless(poster.size > 0)

        def request_done(result_model):
            for movie in result_model.movies:
                if movie.posters:
                    keys = movie.posters.keys()
                    first = movie.posters[keys[0]][0]
                    model, dfr = self.provider.get(first)
                    dfr.addCallback(got_poster)
                    return dfr

            self.fail("No poster found")

        dfr = self._search_terminator()
        dfr.addCallback(request_done)
        return dfr
