# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

import mimetypes

class UnknownFileExtension(Exception):
    """
    Raised by the L{typefind} function when it doesn't recognize a
    file extension.
    """

    def __init__(self, filename):
        self.filename = filename

    def __repr__(self):
        return "<UnknownFileExtension filename=%r>" % self.filename

def typefind(uri):
    """
    Retrieve the media type of the given URI, based on its
    extension.

    Returns ['audio',] if the uri is an audio file
    Returns ['video',] if the uri is a video file
    Returns ['image',] if the uri is an image file
    Returns ['audio','video'] if the uri extension is ogg

    If the extension is not recognized, raise L{UnknownFileExtension}

    @param uri: the media resource address to analyze
    @type uri:  L{elisa.core.media_uri.MediaUri}
    @rtype:     list of strings
    """
    mime_type = mimetypes.guess_type(uri.path)
    if mime_type[0]:
        parts = mime_type[0].split('/')
        mime_type = mime_type[0]
        # Special cases handling
        if parts[0] == 'application':
            ext = uri.extension
            custom_extensions = {'ogg': ['audio','video'],
                                 'iso': 'video',
                                 'mkv': 'video',
                                }
            file_type = custom_extensions.get(ext, '')
        else:
            file_type = parts[0]
    else:
        # in case mimetypes did not return anything, filter on extension
        custom_extensions = {'flv': 'video',
                             'ogm': 'video',
                             'mkv': 'video',
                             'mod': 'video',
                             'nef': 'image',
                             'ogg': ['audio', 'video'],
                             'ogv': 'video',
                             'oga': 'audio',
                             'wmv': 'video',
                             'wma': 'audio',
                             'asf': 'video',
                             'mp4': 'video',
                             'ts':  'video',
                             'm2v': 'video',
                             'swf': 'video',
                             'vob': 'video',
                             'ram': 'audio',
                             'm4a': 'audio',
                             'iso': 'video',
                             'divx': 'video',
                             'flac': 'audio',
                             }
        file_type = custom_extensions.get(uri.extension)
    if not isinstance(file_type, (list, tuple)):
        file_type = [file_type,]
    if not file_type or not set(file_type).intersection(('audio', 'image', 'video')):
        raise UnknownFileExtension(uri.path)
    return file_type
