/*  
  edit.c  -- editing module for an interactive editor of prerequisite-chart 
             descriptions
  Copyright (c) 2005-10  R. D. Tennent   
  School of Computing, Queen's University, rdt@cs.queensu.ca 

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

# include "prerex.h"

PRIVATE element *first_cut;		/* stack of cut nodes */
PRIVATE element *first_deletion;	/* stack of deleted nodes/arrows */

PRIVATE char *command_line;
PRIVATE char *clp;

PRIVATE void
free_elements (element ** pb)
{
  if (*pb != NULL)
    {
      element *pbb = *pb;
      free_elements (&(pbb->next));
      free (pbb);
      *pb = NULL;
    }
}

PRIVATE void
free_lists (void)
{
  free_elements (&first_node);
  free_elements (&first_arrow);
  free_elements (&first_cut);
  free_elements (&first_deletion);
}

PRIVATE void
execute_shell_command (void)
{
  fclose (tex_file);
  restore_write_access ();
  if (system (clp)) puts("System call failed.");
  tex_file = fopen (chartfilename, "r+");
  if (tex_file == NULL) error ("Can't re-open the .tex file.");
  remove_write_access ();
  close_files(); 
  free_lists();
  analyze_tex_file ();
  regenerate_and_process ();
}

PRIVATE void
shiftnodes ( element *pb, coord c )
{
  while (pb != NULL)
    {
      pb->u.n.at.x += c;
      pb = pb->next;
    }
}

PRIVATE void
shift (void)
{
  coord c;
  element *pb = NULL;
  if (sscanf (command_line, "%*s %i", &c) != 1)
    {
      puts ("Can't read shift amount.");
      return;
    }
  clp = command_line;
  while (isspace (*clp)) clp++;
  while (isalpha (*clp)) clp++;  /* command */
  while (isspace (*clp)) clp++;
  if (*clp == '-' || *clp == '+') clp++;
  while (isdigit (*clp)) clp++;   /* shift amount */
  while (true) /* process coordinate pairs */
  {
    point p;
    if (sscanf (clp, "%i", &p.x) != 1) break;
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    while (isspace (*clp)) clp++;
    if (*clp != ',') {
      puts("Can't analyze coordinates.");
      regenerate_and_process ();  /* some nodes may already have been shifted */
      return;
    }
    clp++;  /* ','  */
    if (sscanf (clp, "%i", &p.y) != 1) {
      puts("Can't analyze coordinates.");
      regenerate_and_process ();  /* some nodes may already have been shifted */
      return;
    }
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    pb = node_at (p);
    if (pb == NULL) 
    {
       printf("No node at coordinates %i,%i.\n", p.x, p.y);
       regenerate_and_process ();  /* some nodes may already have been shifted */
       return;
    }
    pb->u.n.at.x += c;
  }
  if (pb == NULL)  /* shift everything */
  {
    shiftnodes(first_node, c);
    shiftnodes(first_cut, c);
    shiftnodes(first_deletion, c);
    /* arrows refer to source and target nodes and will be shifted automatically.  */
  }
  regenerate_and_process ();
}

PRIVATE void
raisenodes ( element *pb, coord c )
{
  while (pb != NULL)
    {
      pb->u.n.at.y += c;
      pb = pb->next;
    }
}

PRIVATE void
Raise (void)  /* R to avoid conflicting with signal.h function raise */
{
  element *pb = NULL;
  coord c;
  if (sscanf (command_line, "%*s %i", &c) != 1)
    {
      puts ("Can't read raise amount.");
      return;
    }
  clp = command_line;
  while (isspace (*clp)) clp++;
  while (isalpha (*clp)) clp++;  /* command */
  while (isspace (*clp)) clp++;
  if (*clp == '-' || *clp == '+') clp++;
  while (isdigit (*clp)) clp++;   /* raise amount */
  while (true) /* process coordinate pairs */
  {
    point p;
    if (sscanf (clp, "%i", &p.x) != 1) break;
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    while (isspace (*clp)) clp++;
    if (*clp != ',') {
      puts("Can't analyze coordinates.");
      regenerate_and_process ();  /* some nodes may already have been raised */
      return;
    }
    clp++;  /* ','  */
    if (sscanf (clp, "%i", &p.y) != 1) {
      puts("Can't analyze coordinates.");
      regenerate_and_process ();  /* some nodes may already have been raised */
      return;
    }
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    pb = node_at (p);
    if (pb == NULL) 
    {
       printf("No node at coordinates %i,%i.\n", p.x, p.y);
       regenerate_and_process ();  /* some nodes may already have been shifted */
       return;
    }
    pb->u.n.at.y += c;
  }
  if (pb == NULL)  /* raise everything */
  {
    raisenodes(first_node, c);
    raisenodes(first_cut, c);
    raisenodes(first_deletion, c);
    /* arrows refer to source and target nodes and will be raised automatically.  */
  }
  regenerate_and_process ();
}

PRIVATE void
cut_node (point p)
{
  element *pn, *pnt;
  pnt = NULL;
  pn = first_node;
  while (pn != NULL && !eq (p, pn->u.n.at))
    {
      pnt = pn;
      pn = pn->next;
    }
  if (pn == NULL)
    {
      puts ("No course box, mini or text at this location.");
      return;
    }
  /* move node to cut list */
  if (pnt == NULL)
    first_node = pn->next;
  else
    pnt->next = pn->next;
  pn->next = first_cut;
  first_cut = pn;
}

PRIVATE void
delete_node (point p)
{
  element *pn, *pnt;
  pnt = NULL;
  pn = first_node;
  while (pn != NULL && !eq (p, pn->u.n.at))
    {
      pnt = pn;
      pn = pn->next;
    }
  if (pn == NULL)
    {
      puts ("No course box, mini or text at this location.");
      return;
    }
  /* move node to deletion list */
  if (pnt == NULL)
    first_node = pn->next;
  else
    pnt->next = pn->next;
  pn->next = first_deletion;
  first_deletion = pn;
  printf("Deleted node at %i,%i.\n", p.x, p.y);
}

PRIVATE void
delete_arrow (point p0, point p1)
{
  element *n0 = node_at (p0);
  element *n1 = node_at (p1);
  element *pa;
  element *pt; 
  if (n0 == NULL)
    {
      puts ("No course box, mini or text at the source location.");
      return;
    }
  if (n1 == NULL)
    {
      puts ("No course box, mini or text at the target location.");
      return;
    }
  pa = first_arrow;
  pt = NULL;		/* trailing pointer */
  while (pa != NULL && (pa->u.a.source != n0 || pa->u.a.target != n1))
    {
      pt = pa;
      pa = pa->next;
    }
  if (pa == NULL)
    {
      puts ("No such arrow.");
      return;
    }
  if (pt == NULL)
    first_arrow = pa->next;
  else
    pt->next = pa->next;
  pa->next = first_deletion;
  first_deletion = pa;
  printf("Deleted arrow from %i,%i to %i,%i.\n", p0.x, p0.y, p1.x, p1.y);
}

PRIVATE void
analyze_cut_command (void)
{
  point p, q;
  if (sscanf (command_line, "%*s %i,%i,%i,%i", &p.x, &p.y, &q.x, &q.y) == 4)
    {
      puts ("Can't cut arrows; deleting instead.");
      delete_arrow (p, q);
      regenerate_and_process ();
    }
  else if (sscanf (command_line, "%*s %i,%i", &p.x, &p.y) == 2)
    {
      cut_node (p);
    }
  else
    {
      puts ("Can't analyze cut command.");
      return;
    }
}

PRIVATE void
analyze_delete_command (void)
{
  clp = command_line;
  while (isspace (*clp)) clp++;
  while (isalpha (*clp)) clp++;  /* command */
  while (isspace (*clp)) clp++;
  while (true) /* process coordinate pairs/4-tuples */
  {
    point p0, p1; 
    element *pb;
    if (sscanf (clp, "%i", &p0.x) != 1) break;
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    while (isspace (*clp)) clp++;
    if (*clp != ',') {
      puts("Can't analyze coordinates.");
      break;
    }
    clp++;  /* ','  */
    if (sscanf (clp, "%i", &p0.y) != 1) {
      puts("Can't analyze coordinates.");
      break;
    }
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    pb = node_at (p0);
    if (pb == NULL) 
    {
       printf("No node at coordinates %i,%i.\n", p0.x, p0.y);
       continue;
    }
    while (isspace (*clp)) clp++;
    if (*clp != ',') 
    {
      delete_node (p0);
      continue;
    }
    clp++;  /* ','  */
    if (sscanf (clp, "%i", &p1.x) != 1) {
      puts("Can't analyze coordinates.");
      break;
    }
    while (isspace (*clp)) clp++;
    if (*clp == '-' || *clp == '+') clp++;
    while (isdigit(*clp)) clp++;
    while (isspace (*clp)) clp++;
    if (*clp != ',') {
      puts("Can't analyze coordinates.");
      break;
    }
    clp++;  /* ','  */
    if (sscanf (clp, "%i", &p1.y) != 1) {
      puts("Can't analyze coordinates.");
      break;
    }
    while (isdigit(*clp)) clp++;
    pb = node_at (p1);
    if (pb == NULL) 
    {
       printf("No node at coordinates %i,%i.\n", p1.x, p1.y);
       continue;
    }
    delete_arrow (p0, p1);
  }
  regenerate_and_process ();
}

PRIVATE void
undelete (void)
{
  element *pn = first_deletion;
  if (pn == NULL)
    {
      puts ("Nothing to undelete.");
      return;
    }
  first_deletion = pn->next;
  if (pn->tag == NODE)
    {
      if (insert_node (pn))
	{
	  puts ("There is now another course box, mini or text at that location.");
	  pn->next = first_deletion;
	  first_deletion = pn;
	  return;
	}
      printf ("Course box, mini or text at %i,%i undeleted.\n",
	      pn->u.n.at.x, pn->u.n.at.y);
    }
  else if (pn->tag == ARROW)
    {
      if (insert_arrow (pn))
	{
	  puts
	    ("Can't undelete arrow: source and/or target is/are now missing.");
	  return;
	}
      printf ("Arrow from %i,%i to %i,%i undeleted.\n",
	      pn->u.a.source->u.n.at.x, pn->u.a.source->u.n.at.y,
	      pn->u.a.target->u.n.at.x, pn->u.a.target->u.n.at.y);
    }
  else
    {
      puts ("Undefined deletion type.");
      return;
    }
  regenerate_and_process ();
}

PRIVATE void
edit_mini (element * pm)
{
  deftext[0] = '\0';
  append (deftext, NULL, pm->u.n.code, sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("course code: ");
  pm->u.n.code[0] = '\0';
  if (append (pm->u.n.code, NULL, command_line, sizeof (pm->u.n.code))
      >= sizeof (pm->u.n.code))
    puts ("Warning: course code too long, truncated.");
}

PRIVATE void
edit_text (element * pm)
{
  deftext[0] = '\0';
  append (deftext, NULL, pm->u.n.u.t.txt, sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("text: ");
  pm->u.n.u.t.txt[0] = '\0';
  if (append (pm->u.n.u.t.txt, NULL, command_line, sizeof (pm->u.n.u.t.txt))
      >= sizeof (pm->u.n.u.t.txt))
    puts ("Warning: text too long, truncated.");
}

PRIVATE void
edit_box (element * pb)
{
  char code[8];
  deftext[0] = '\0';
  append (deftext, NULL, pb->u.n.code, sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("course code: ");
  pb->u.n.code[0] = '\0';
  if (append (pb->u.n.code, NULL, command_line, sizeof (pb->u.n.code))
      >= sizeof (pb->u.n.code))
    puts ("Warning: course code too long, truncated.");
  deftext[0] = '\0';
  if (pb->u.n.u.b.half)
    append (deftext, NULL, "y", sizeof (deftext));
  else
    append (deftext, NULL, "n", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("half course (y/n)? ");
  sscanf (command_line, "%1s", code);
  if (code[0] == 'y')
    pb->u.n.u.b.half = true;
  else if (code[0] == 'n')
    pb->u.n.u.b.half = false;
  else
    {
      puts ("Response not recognized; 'y' assumed.");
      pb->u.n.u.b.half = true;
    }
  deftext[0] = '\0';
  if (pb->u.n.u.b.required)
    append (deftext, NULL, "y", sizeof (deftext));
  else
    append (deftext, NULL, "n", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("required (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] == 'y')
    pb->u.n.u.b.required = true;
  else if (code[0] == 'n')
    pb->u.n.u.b.required = false;
  else
    {
      puts ("Response not recognized; 'n' assumed.");
      pb->u.n.u.b.half = true;
    }
  deftext[0] = '\0';
  append (deftext, NULL, pb->u.n.u.b.title, sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("course title: ");
  pb->u.n.u.b.title[0] = '\0';
  if (append (pb->u.n.u.b.title, NULL, command_line, sizeof (pb->u.n.u.b.title))
      >= sizeof (pb->u.n.u.b.title))
    puts ("Warning: course title too long, truncated.");
  deftext[0] = '\0';
  append (deftext, NULL, pb->u.n.u.b.timetable, sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("course timetable: ");
  pb->u.n.u.b.timetable[0] = '\0';
  if (append
      (pb->u.n.u.b.timetable, NULL, command_line,
       sizeof (pb->u.n.u.b.timetable)) >= sizeof (pb->u.n.u.b.timetable))
    puts ("Warning: course timetable too long, truncated.");
}

PRIVATE void
set_curvature (element * pa)
{
  char code[16];
  deftext[0] = '\0';
  if (pa->u.a.curvature >= 0 && pa->u.a.curvature <= 100)
    {
      sprintf (deftext, "%i", pa->u.a.curvature);
      rl_startup_hook = set_deftext;
    }
  else
    {
      append (deftext, NULL, "d", sizeof (deftext));
      rl_startup_hook = set_deftext;
    }
  free (command_line);
  command_line = readline ("curvature, default (d) or int value? ");
  sscanf (command_line, "%15s", code);
  if (isdigit (code[0]))
    {
      sscanf (code, "%i", &(pa->u.a.curvature));
      if (pa->u.a.curvature > 100) {
        puts ("Curvature too large; default value used.");
        pa->u.a.curvature = -1;
      }
    }
  else if (code[0] == 'd')
    pa->u.a.curvature = -1;	/* negative value denotes default curvature */
  else
    {
      puts ("Response not recognized; default assumed.");
      pa->u.a.curvature = -1;
    }
}

PRIVATE void
edit_arrow (element * pa)
{
  char code[16];
  deftext[0] = '\0';
  switch (pa->u.a.tag)
    {
    case PREREQ:
      append (deftext, NULL, "p", sizeof (deftext));
      break;
    case COREQ:
      append (deftext, NULL, "c", sizeof (deftext));
      break;
    case RECOMM:
      append (deftext, NULL, "r", sizeof (deftext));
      break;
    default:;
    }
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line =
    readline ("prerequisite (p), corequisite (c), or recommended (r)? ");
  sscanf (command_line, "%15s", code);
  switch (code[0])
    {
    case 'p':
      pa->u.a.tag = PREREQ;
      break;
    case 'c':
      pa->u.a.tag = COREQ;
      break;
    case 'r':
      pa->u.a.tag = RECOMM;
      break;
    default:
      puts ("Response not recognized; prerequisite assumed.");
      pa->u.a.tag = PREREQ;
    }
  set_curvature (pa);
}

PRIVATE void
analyze_paste_command (void)
{
  point p;
  if (sscanf (command_line, "%*s %i,%i", &p.x, &p.y) == 2)
    {
      element *pn = first_cut;
      if (pn == NULL)
	{
	  puts ("No cut box, mini or text to paste.");
	  return;
	}
      first_cut = pn->next;
      pn->u.n.at = p;
      if (insert_node (pn))
	{
	  puts ("There is a course box, mini, or text already at that location.");
	  pn->next = first_cut;
	  first_cut = pn;
	  return;
	}
    }
  else
    {
      puts ("Can't analyze paste command.");
      return;
    }
  if (first_cut == NULL)
    regenerate_and_process ();
}

PRIVATE void
analyze_box_command (void)
{
  point p;
  element *pb;
  char code[8];
  if (sscanf (command_line, "%*s %i,%i", &p.x, &p.y) != 2)
    {
      puts ("Can't analyze box command.");
      return;
    }
  pb = node_at (p);
  if (pb != NULL)
    {
      edit_box (pb);
      regenerate_and_process ();
      return;
    }
  deftext[0] = '\0';
  append (deftext, NULL, "y", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("create new course box (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] == 'y')
    {
      pb = (element *) malloc (sizeof (element));
      if (pb == NULL)
	error ("Out of memory");
      pb->tag = NODE;
      pb->u.n.tag = BOX;
      pb->u.n.at = p;
      pb->u.n.u.b.required = false;
      pb->u.n.u.b.half = true;
      pb->u.n.code[0] = '\0';
      pb->u.n.u.b.title[0] = '\0';
      pb->u.n.u.b.timetable[0] = '\0';
      if (insert_node (pb))
	{
	  puts ("There is already a course box, mini, or text at that location.");
          free (pb);
	  return;
	}
      edit_box (pb);
    }
  else if (code[0] == 'n')
    return;
  else
    {
      puts ("Response not recognized; 'n' assumed.");
      return;
    }
  regenerate_and_process ();
}

PRIVATE void
analyze_mini_command (void)
{
  point p;
  element *pm;
  char code[8];
  if (sscanf (command_line, "%*s %i,%i", &p.x, &p.y) != 2)
    {
      puts ("Can't analyze mini command.");
      return;
    }
  pm = node_at (p);
  if (pm != NULL)
    {
      edit_mini (pm);
      regenerate_and_process ();
      return;
    }
  deftext[0] = '\0';
  append (deftext, NULL, "y", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("create new mini (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] == 'y')
    {
      pm = (element *) malloc (sizeof (element));
      if (pm == NULL)
	error ("Out of memory");
      pm->tag = NODE;
      pm->u.n.tag = MINI;
      pm->u.n.at = p;
      pm->u.n.code[0] = '\0';
      if (insert_node (pm))
	{
	  puts ("There is already a course box, mini, or text at that location.");
          free (pm);
	  return;
	}
      edit_mini (pm);
    }
  else if (code[0] == 'n')
    return;
  else
    {
      puts ("Response not recognized; 'n' assumed.");
      return;
    }
  regenerate_and_process ();
}

PRIVATE void
analyze_text_command (void)
{
  point p;
  element *pm;
  char code[8];
  if (sscanf (command_line, "%*s %i,%i", &p.x, &p.y) != 2)
    {
      puts ("Can't analyze text command.");
      return;
    }
  pm = node_at (p);
  if (pm != NULL)
    {
      edit_text (pm);
      regenerate_and_process ();
      return;
    }
  deftext[0] = '\0';
  append (deftext, NULL, "y", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("create new text (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] == 'y')
    {
      pm = (element *) malloc (sizeof (element));
      if (pm == NULL)
	error ("Out of memory");
      pm->tag = NODE;
      pm->u.n.tag = TEXT;
      pm->u.n.at = p;
      pm->u.n.code[0] = '\0';
      if (insert_node (pm))
	{
	  puts ("There is already a course box, mini, or text at that location.");
          free (pm);
	  return;
	}
      edit_text (pm);
    }
  else if (code[0] == 'n')
    return;
  else
    {
      puts ("Response not recognized; 'n' assumed.");
      return;
    }
  regenerate_and_process ();
}

PRIVATE void
analyze_arrow_command (void)
{
  point p0, p1;
  element *n0;
  element *n1 = node_at (p1);
  element *pa = first_arrow;
  if (sscanf (command_line, "%*s %i,%i,%i,%i", &p0.x, &p0.y, &p1.x, &p1.y) !=
      4)
    {
      puts ("Can't analyze arrow command.");
      return;
    }
  n0 = node_at (p0);
  if (n0 == NULL)
    {
      puts ("There is no course box, mini, or text at the source location.");
      return;
    }
  n1 = node_at (p1);
  if (n1 == NULL)
    {
      puts ("There is no course box, mini, or text at the target location.");
      return;
    }
  while (pa != NULL && !(pa->u.a.source == n0 && pa->u.a.target == n1))
    pa = pa->next;
  if (pa != NULL)
    edit_arrow (pa);
  else
    {
      char code[8];
      deftext[0] = '\0';
      append (deftext, NULL, "y", sizeof (deftext));
      rl_startup_hook = set_deftext;
      free (command_line);
      command_line = readline ("create new arrow (y/n)? ");
      sscanf (command_line, "%7s", code);
      if (code[0] == 'y')
	{
	  pa = (element *) malloc (sizeof (element));
	  if (pa == NULL)
	    error ("Out of memory");
	  pa->tag = ARROW;
	  pa->u.a.source = n0;
	  pa->u.a.target = n1;
	  pa->u.a.tag = PREREQ;
	  pa->u.a.curvature = -1;	/* default curvature */
	  insert_arrow (pa);
	  edit_arrow (pa);
	}
      else if (code[0] == 'n')
	return;
      else
	{
	  puts ("Response not recognized; 'n' assumed.");
	  return;
	}
    }
  regenerate_and_process ();
}

PRIVATE void
analyze_grid_command (void)
{
  char code[8];
  deftext[0] = '\0';
  if (sscanf (command_line, "%*s %7s", code) != 1)
    {
      if (grid)
	append (deftext, NULL, "n", sizeof (deftext));
      else
	append (deftext, NULL, "y", sizeof (deftext));
      rl_startup_hook = set_deftext;
      free (command_line);
      command_line = readline ("grid (y/n)? ");
      sscanf (command_line, "%7s", code);
    }
  if (code[0] == 'y')
    grid = true;
  else if (code[0] == 'n')
    grid = false;
  else
    {
      puts ("Response not recognized; 'y' assumed.");
      grid = true;
    }
  regenerate_and_process ();
}

PRIVATE void
analyze_file_command (void)
{
  puts ("This command is no longer supported.");
}

PRIVATE void
backup (void)
{
  char code[8];
  deftext[0] = '\0';
  append (deftext, NULL, "y", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line = readline ("Overwrite current backup file (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] != 'y')
    return;
  backup_tex_file = fopen (backup_filename, "w");
  if (backup_tex_file == NULL)
    {
      puts ("Can't open backup file.");
      return;
    }
  copy (tex_file, backup_tex_file);
  fclose (backup_tex_file);
  printf ("Backed up to %s.\n", backup_filename);
}

PRIVATE void
restore (void)
{
  char code[8];
  backup_tex_file = fopen (backup_filename, "r");
  if (backup_tex_file == NULL)
    {
      puts ("Can't open backup file.");
      return;
    }
  deftext[0] = '\0';
  append (deftext, NULL, "y", sizeof (deftext));
  rl_startup_hook = set_deftext;
  free (command_line);
  command_line =
    readline ("Delete current TeX file and restore from backup (y/n)? ");
  sscanf (command_line, "%7s", code);
  if (code[0] != 'y')
    return;
  restore_write_access ();
  tex_file = fopen (chartfilename, "w+");
  rewind (tex_file); 
  copy (backup_tex_file, tex_file);
  remove_write_access ();
  fclose (backup_tex_file);
  close_files();
  free_lists();
  grid = false;
  analyze_tex_file ();
  regenerate_and_process ();
}


PRIVATE void
quit (void)
{
  char code[8];
  deftext[0] = '\0';
  if (first_cut)
    {
      append (deftext, NULL, "y", sizeof (deftext));
      rl_startup_hook = set_deftext;
      free (command_line);
      command_line =
        readline ("Warning: there are unpasted cuts; continue (y/n)? ");
      sscanf (command_line, "%7s", code);
      if (code[0] != 'y')
        return;
    }
  grid = false;
  puts("Turning off coordinate grid.");
  regenerate_and_process ();
  restore_write_access ();
  exit (0);
}

PRIVATE void
help (void)
{
  puts ("");
  puts ("        command:                effect:");
  puts ("");
  puts (" file> box    x,y               [create and] edit course box at x,y");
  puts (" file> mini   x,y               [create and] edit mini course at x,y");
  puts (" file> text   x,y               [create and] edit text centered at x,y");
  puts (" file> arrow  x0,y0,x1,y1       [create and] edit arrow from x0,y0 to x1,y1");
  puts (" file> cut    x,y               (temporarily) remove box, mini, or text at x,y");
  puts (" file> paste  x,y               re-insert removed box, mini, or text at x,y");
  puts (" file> delete x0,y0[,x1,y1] ... remove specified elements/arrows");
  puts (" file> undelete                 undelete deleted box, mini, text, or arrow");
  puts (" file> shift [-]nx [xi,yi ...]  move [specified] elements nx units right [left]");
  puts (" file> raise [-]ny [xi,yi ...]  move [specified] elements ny units up [down]");
  puts (" file> ! | write                save to file.tex and process (with pdflatex)");
  puts (" file> quit | exit | x | ^D     turn off grid, save, process, and exit");
  puts (" file> !cmd                     execute shell command cmd, then re-load");
  puts (" file> Backup                   copy file.tex to .file.tex");
  puts (" file> Restore                  restore from .file.tex");
  puts (" file> grid   [y/n]             turn on/off coordinate-grid background");
  puts (" file> help | ?                 print this summary");
  puts ("");
  puts ("Append \";\" to commands to suppress usual save-and-process.");
  puts("");
}


PRIVATE void
analyze_user_command (void)
{
  /* free(command_line) just *before* any additional calls to readline */

  char command[COMMAND_LEN + 1] = {'\0'};
  sscanf (command_line, "%63s", command);

  if (prefix (command, "write"))
    {
      reprocess = true;
      regenerate_and_process ();
    }
  else if (prefix (command, "shift"))
    shift ();
  else if (prefix (command, "raise"))
    Raise ();
  else if (prefix (command, "cut"))
    analyze_cut_command ();
  else if (prefix (command, "delete"))
    analyze_delete_command ();
  else if (prefix (command, "undelete"))
    undelete ();
  else if (prefix (command, "paste"))
    analyze_paste_command ();
  else if (prefix (command, "box"))
    analyze_box_command ();
  else if (prefix (command, "mini"))
    analyze_mini_command ();
  else if (prefix (command, "text"))
    analyze_text_command ();
  else if (prefix (command, "arrow"))
    analyze_arrow_command ();
  else if (prefix (command, "quit")
	   || prefix (command, "exit") || prefix (command, "x"))
    {
      quit ();
    }
  else if (prefix (command, "grid"))
    analyze_grid_command ();
  else if (prefix (command, "file"))
    analyze_file_command ();
  else if (command[0] == '!')
    {
      clp = command_line;
      while (isspace (*clp))
	clp++;
      clp++;			/* ! */
      while (isspace (*clp))
	clp++;
      if (*clp == '\0')
	{
          reprocess = true;
	  regenerate_and_process ();
	}
      else
	{
	  execute_shell_command ();
	}
    }
  else if (prefix (command, "Backup"))
    backup ();
  else if (prefix (command, "Restore"))
    restore ();
  else if (prefix (command, "help"))
    help ();
  else if (command[0] == '?')
    help ();
  else
    {
      puts ("Command not recognized.");
      help ();
    }
}

void process_commands(void)
{
  char prompt[FILE_LEN + 8] = {'\0'};
  char *prompt_n = prompt;
  help ();
  do
    {
      prompt[0] = '\0';
      prompt_n = prompt;
      append (prompt, &prompt_n, chartfilename, sizeof (prompt));
      prompt_n = prompt_n - 4; *prompt_n = '\0'; /* suppress ".tex"  */
      append (prompt, &prompt_n, "> ", sizeof (prompt));
      command_line = readline (prompt);
      if (command_line == NULL)    /* EOF */
      { putchar ('\n');
	quit();		
      }
      if (command_line[0] != '\0')
	{
	  add_history (command_line);
          reprocess = true;
          if (suffix(";", command_line))
          {
            /* suppress possible regeneration and processing after this command */
            reprocess = false;
            command_line[strlen(command_line)-1] = '\0';
          }
	  analyze_user_command ();
	}
      free (command_line);
    }
  while (true);			/* exit via call to exit or error()  */
}
