/*
 * Copyright 2005 - 2009  Zarafa B.V.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/**
* This class is managing keyboard- and mouse input from the user and sending these
* events to the correct module by keeping track of the "focus"
*/
function InputManager()
{
	this.objects = new Object();
	this.objectCount = 0;
	this.focusId = false;
	this.oldFocusId = false;
this.blurStatus = false;

	this.bindings = {
					"mouseup": new Array(), 
					"mousedown": new Array(), 
					"mousemove": new Array(), 
					"contextmenu": new Array(), 
					"keydown": new Array(), 
					"keyup": new Array(),
					"focus": new Array(), // focus & blur events are generated by the inputmanager!
					"blur" : new Array()
					};
					
	this.keyControlObjects = new Object();
	this.keyControlBindings = {
								"keydown": new Array(), 
								"keyup": new Array()
							  };
	this.keyStrokes = new Array();
}


InputManager.prototype.init = function()
{
	// register several events we want to handle
	dhtml.addEvent(this, document.body, "mouseup", eventInputManager);
	dhtml.addEvent(this, document.body, "mousedown", eventInputManager);
	dhtml.addEvent(this, document.body, "mousemove", eventInputManager);
	dhtml.addEvent(this, document.body, "contextmenu", eventInputManager);
	dhtml.addEvent(this, document, "keydown", eventInputManager);
	dhtml.addEvent(this, document, "keyup", eventInputManager);
	dhtml.addEvent(this, document, "keypress", eventInputManager);

	// register the webclient object, which can be used by anyone who want to bind an event but didn't have a object.
	//this.addObject(webclient, dhtml.getElementById("main"));
	//this.addObject(webclient, dhtml.getElementById("dialog_content"));
	this.addObject(webclient, document.body);
	this.defaultId = this.getObjectId(webclient);

	// add focus handling
	dhtml.addEvent(this, document, "focus", eventInputManagerFocusControl);
	dhtml.addEvent(this, document, "blur", eventInputManagerFocusControl);
	dhtml.addEvent(this, document.body, "mousedown", eventInputManagerFocusControl);

	this.resetTimeout();
	this.initKeyControl();
}

InputManager.prototype.resetTimeout = function()
{
	if (this.timeoutTimer){
		window.clearTimeout(this.timeoutTimer);
		this.timeoutTimer = false;
	}
	
	if (typeof webclient != "undefined"){
		var timer = webclient.settings.get("global/auto_logout", 0);
		if (timer>0){
			this.timeoutTimer = window.setTimeout(eventInputTimerLogout, timer);
		}
	}
}

/**
* Use this function to register your module, the focusElement is the main container of the module
*/
InputManager.prototype.addObject = function(newObject, focusElement)
{
	if (!focusElement) {
		focusElement = false;
	}
	this.objects[this.objectCount++] = {"object":newObject,"element":focusElement};
}

InputManager.prototype.removeObject = function(object)
{
	var objectId = this.getObjectId(object);

	while(objectId){
		// first remove all events for this object
		for(var eventType in this.bindings){
			for(var i=0; i<this.bindings[eventType].length; i++){
				var binding = this.bindings[eventType][i];
				if (binding && binding.objectId == objectId){
					this.bindings[eventType].splice(i,1);
					i--;
				}
			}
		}

		// remove all keycontrol events for this object
		for(var eventType in this.keyControlBindings){
			for(var i=0; i<this.keyControlBindings[eventType].length; i++){
				var binding = this.keyControlBindings[eventType][i];
				if (binding && binding.objectId == objectId){
					this.keyControlBindings[eventType].splice(i,1);
					i--;
				}
			}
		}

		// next delete the object itself
		delete this.objects[objectId];
		// Same object might be registered more than one time, so retrieve new objectId of same object
		objectId = this.getObjectId(object);
	}
}

/*
* Checks if the given module is selected
*/
InputManager.prototype.hasFocus = function(object)
{
	var result = false;
	if (this.focusId!=false && typeof this.objects[this.focusId] != "undefined"){
		result = (object==this.objects[this.focusId]["object"]);
	}
	return result;
}

/*
* Internal function to keep track of focus
*/
InputManager.prototype.handleFocus = function(event)
{
	this.resetTimeout();

	switch(event.type)
	{
		case "blur": // other window/iframe
			this.oldFocusId = this.focusId;
			this.focusId = false;
			this.blurStatus = true;
			break;
		case "focus":
			if(this.blurStatus){
				this.focusId = this.oldFocusId;
				this.blurStatus = false;
			}
			break;
		case "mousedown": // manual focus
			
			var element = event.target;
			
			if (this.focusId === false){
				window.focus();
			}
			
			this.changeFocus(element);
			break;
	}
	return true;
}

InputManager.prototype.changeFocus = function(element){
	// Find the object that should get the focus
	var rootElement = element;
	var objectId = false;
	while(rootElement!=null){
		objectId = this.getObjectIdByElement(rootElement);
		if (objectId===false){
			rootElement = rootElement.parentNode;
		}else{
			break;
		}
	}

	/**
	 * If previewpane is there then there is a iframe,
	 * so on focus change event first we have to blur the iframe
	 * Then set focus on the window and selected element.
	 * Iframe window events won't be bubbled to main WA widow.
	 */
	var htmlbody = dhtml.getElementById("html_body");
	if(htmlbody) {
		htmlbody.blur();
	}
	// Change focus
	this.focusId = objectId;
	// Focus on the new object
	this.handleEvent(element, {type: "focus"});
}


InputManager.prototype.getObjectIdByElement = function(element)
{
	for(var i in this.objects){
		if (this.objects[i]["element"] == element){
			return i;
		}
	}
	return false;
}

InputManager.prototype.getObjectId = function(object)
{
	for(var i in this.objects){
		if (this.objects[i]["object"] == object){
			return i;
		}
	}
	return false;
}

/**
* After you registered your module, you can use this function to bind some events to your module
*/
InputManager.prototype.bindEvent = function(object, eventType, handler)
{
	if (typeof this.bindings[eventType] == "undefined"){
		return false;
	}

	var binding = new Object();
	binding.objectId = this.getObjectId(object);
	binding.handler = handler;

	this.bindings[eventType].push(binding);
	return true;
}

InputManager.prototype.unbindEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.bindings[eventType]){
		var binding = this.bindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			this.bindings[eventType].splice(i,1);
			return true;
		}
	}
	return false;
}

InputManager.prototype.handleEvent = function(element, event)
{
	this.resetTimeout();
	var returnValue = true;

	// Create the inputmanger event
	var imEvent = new Object();
	imEvent.altKey   = event.altKey;
	imEvent.charCode = event.charCode;
	imEvent.ctrlKey  = event.ctrlKey;
	imEvent.isChar   = event.isChar;
	imEvent.keyCode  = event.keyCode;
	imEvent.metaKey  = event.metaKey;
	imEvent.shiftKey = event.shiftKey;
	imEvent.type     = event.type;
	imEvent.which    = event.which;
	imEvent.clientX  = event.clientX;
	imEvent.clientY  = event.clientY;
	imEvent.hideContextMenu  = event.hideContextMenu;
//TODO
// Consider what needs be in target and what needs to be in element. 
// The function checkMenuState seems to be added by the webclient through the InputManager.
// The element argument is the body (regsitered element for the webclient object)
// The event.target (or event.srcElement: check IE) is used to get the actual clicked element.
	imEvent.target   = event.target;

	for(var i in this.bindings[event.type]) {
		var binding = this.bindings[event.type][i];
		if (binding && binding.objectId && (binding.objectId == this.defaultId || binding.objectId == this.focusId)){
			if (binding.handler(this.objects[binding.objectId]["object"], element, imEvent) === false)
				returnValue = false;
		}
	}
	return returnValue;
}

/**
 * Function which tells whether event has 
 * already been binded or not.
 * @param object object 	module object
 * @param string eventType 	type of event
 * @param object handler 	event handler
 * @return boolean 			true if event is already binded, false if not binded
 */
InputManager.prototype.hasEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.bindings[eventType]){
		var binding = this.bindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			return true;
		}
	}
	return false;
}
InputManager.prototype.initKeyControl = function()
{
	dhtml.addEvent(this, document, "keyup", eventInputManagerKeyControlKeyUp);
	dhtml.addEvent(this, document, "keydown", eventInputManagerKeyControlKeyDown);
	
	// Focus handling
	dhtml.addEvent(this, document, "focus", eventInputManagerKeyControlHandleFocus);
	dhtml.addEvent(this, document, "blur", eventInputManagerKeyControlHandleFocus);
}
/**
 * Function which binds an event to object for advanced keycontrol.
 * 
 * @param object object module which is registered to InputManager.
 * @param array key list of key combinations which when pressed should notify module for handling those keys
 * @param string eventType type of event for which object is bind
 * @param object handler event handler
 * @param boolean handleKeysInFocus if true, then only notify module if it has focus, if false then notify without checking focus.
 */ 
InputManager.prototype.bindKeyControlEvent = function(object, keys, eventType, handler, handleKeysInFocus)
{
	if (typeof this.keyControlBindings[eventType] == "undefined"){
		return false;
	}

	var binding = new Object();
	binding.objectId = this.getObjectId(object);
	binding.handler = handler;
	binding.handleKeysInFocus = handleKeysInFocus?handleKeysInFocus:false;
	binding.keys = keys;

	this.keyControlBindings[eventType].push(binding);
	return true;
}
/**
 * Function which unbinds an event from registered object
 *
 * @param object object for which to unbind an event
 * @param string eventType event that is to be unbind
 * @param object handler event handler
 */
InputManager.prototype.unbindKeyControlEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.bindings[eventType]){
		var binding = this.bindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			this.keyControlBindings[eventType].splice(i,1);
			return true;
		}          
	}
	return false;
}
/**
 * Function which notifies module by calling their registered event handler.
 * This function also checks if current key combination should be handled by
 * any of registered module.
 *
 * @param element element on which event has occured
 * @param object event event
 */
InputManager.prototype.handleKeyControlEvent = function(element, event)
{
	// We want to handle keystrokes on input fields only in dialogs.
	var handleKeysOnInput = (typeof parentWebclient != "undefined") ? false : true;
	var bindings = this.keyControlBindings;

	if (event && event.target && (handleKeysOnInput ? checkFieldsForKeycontrol(event) : true)){
		for(var i in bindings[event.type]) {
			var binding = bindings[event.type][i];

			// Check if current key is registered within object.
			if (this.isKeyRegisteredToObject(binding, event.keyCombination)){
				// Check if focus is needed.
				if (binding && binding.objectId && (binding.handleKeysInFocus ? this.hasFocus(this.objects[binding.objectId]["object"]):true)){
					var module = (typeof binding.handleKeysInFocus == "object")?binding.handleKeysInFocus:this.objects[binding.objectId]["object"];

					if (binding.handler.call(module, module, element, event, binding.keys) === false)
						return false;
					else
						return true;
				}
			}
		}
		return true;
	}
}
/**
 * Function which check is current key combination should be handled
 * by specified module.
 * 
 * @param object binding contains module information and list of keys it should handle
 * @param string keyCombination current key combination
 */
InputManager.prototype.isKeyRegisteredToObject = function (binding, keyCombination)
{
	return inArray(keyCombination, binding.keys);
}
/**
 * Function which tells whether event has 
 * already been binded or not.
 * @param object object 	module object
 * @param string eventType 	type of event
 * @param object handler 	event handler
 * @return boolean 			true if event is already binded, false if not binded
 */
InputManager.prototype.hasKeyControlEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.keyControlBindings[eventType]){
		var binding = this.keyControlBindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			return true;
		}
	}
	return false;
}

function eventInputManagerFocusControl(inputmanager, element, event)
{
	return inputmanager.handleFocus(event);
}

function eventInputManager(inputmanager, element, event)
{
	return inputmanager.handleEvent(element, event);
}

function eventInputTimerLogout()
{
	document.location = "index.php?logout=auto";
}
/**
 * Function which handles advance keycontrol
 */
function eventInputManagerKeyControlKeyUp(inputmanager, element, event)
{
	var isKeyControlEnabled = webclient.settings.get("global/shortcuts/enabled", false);

	if (isKeyControlEnabled && !isMenuOpen()){
		// Get key combination
		event.keyCombination = inputmanager.keyStrokes ? inputmanager.keyStrokes.join("+") : '';

		if (typeof inputmanager.keyStrokes != "undefined"){
			for (var i = 0; i < inputmanager.keyStrokes.length; i++){
				if (inputmanager.keyStrokes[i] == event.keyCode.fromCharCode()) inputmanager.keyStrokes.splice(i, 1);
			}
		}
		
		if ((event.keyCode == inputmanager.previousKeyCode) && inputmanager.handleKeyControlEvent(element, event)) {
			if (typeof event.preventDefault != 'undefined') event.preventDefault();
		}

		inputmanager.previousKeyCode = false;
	}
}
/**
 * Function which keeps trac of pressed keys.
 */
function eventInputManagerKeyControlKeyDown(inputmanager, element, event)
{
	var isKeyControlEnabled = webclient.settings.get("global/shortcuts/enabled", false);

	if (isKeyControlEnabled && !isMenuOpen()){
		if (event.keyCode != inputmanager.previousKeyCode){
			inputmanager.keyStrokes.push(event.keyCode.fromCharCode());
			inputmanager.previousKeyCode = event.keyCode;
		}

		/* Check if current keystrokes are either registered by any module.
		 * if registered then return false, because module wants to handle it.
		 */
		var keyCombination = inputmanager.keyStrokes ? inputmanager.keyStrokes.join("+") : '';
		// We want to handle keystrokes on input fields only in dialogs.
		var handleKeysOnInput = (typeof parentWebclient != "undefined") ? false : true;

		if (event && event.target && (handleKeysOnInput ? checkFieldsForKeycontrol(event) : true)){
			for (var type in inputmanager.keyControlBindings){
				for (var i in inputmanager.keyControlBindings[type]){
					var binding = inputmanager.keyControlBindings[type][i];

					if (inputmanager.isKeyRegisteredToObject(binding, keyCombination)) {
						if (typeof event.preventDefault != 'undefined') event.preventDefault();
					}
				}
			}
		}
	}
}
/**
 * Function which initializes key stack on window focus and blur
 */
function eventInputManagerKeyControlHandleFocus(inputmanager, element, event)
{
	inputmanager.keyStrokes = new Array();
}
