# 17 "parsing/lexer.mll"
 
open Lexing
open Misc
open Parser

type error =
  | Illegal_character of char
  | Illegal_escape of string
  | Unterminated_comment
  | Unterminated_string
  | Unterminated_string_in_comment
  | Keyword_as_label of string
  | Literal_overflow of string
;;

exception Error of error * Location.t;;

(* The table of keywords *)

let keyword_table =
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "match", MATCH;
    "method", METHOD;
    "module", MODULE;
    "mutable", MUTABLE;
    "new", NEW;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "private", PRIVATE;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lor", INFIXOP3("lor");
    "lxor", INFIXOP3("lxor");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr")
]

(* To buffer string literals *)

let initial_string_buffer = String.create 256
let string_buff = ref initial_string_buffer
let string_index = ref 0

let reset_string_buffer () =
  string_buff := initial_string_buffer;
  string_index := 0

let store_string_char c =
  if !string_index >= String.length (!string_buff) then begin
    let new_buff = String.create (String.length (!string_buff) * 2) in
      String.blit (!string_buff) 0 new_buff 0 (String.length (!string_buff));
      string_buff := new_buff
  end;
  String.unsafe_set (!string_buff) (!string_index) c;
  incr string_index

let get_stored_string () =
  let s = String.sub (!string_buff) 0 (!string_index) in
  string_buff := initial_string_buffer;
  s

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref Location.none;;
let comment_start_loc = ref [];;
let in_comment () = !comment_start_loc <> [];;

(* To translate escape sequences *)

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let char_for_decimal_code lexbuf i =
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  if (c < 0 || c > 255) then
    if in_comment ()
    then 'x'
    else raise (Error(Illegal_escape (Lexing.lexeme lexbuf),
                      Location.curr lexbuf))
  else Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let d1 = Char.code (Lexing.lexeme_char lexbuf i) in
  let val1 = if d1 >= 97 then d1 - 87
             else if d1 >= 65 then d1 - 55
             else d1 - 48
  in
  let d2 = Char.code (Lexing.lexeme_char lexbuf (i+1)) in
  let val2 = if d2 >= 97 then d2 - 87
             else if d2 >= 65 then d2 - 55
             else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

(* To convert integer literals, allowing max_int + 1 (PR#4210) *)

let cvt_int_literal s =
  - int_of_string ("-" ^ s)
let cvt_int32_literal s =
  Int32.neg (Int32.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_int64_literal s =
  Int64.neg (Int64.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_nativeint_literal s =
  Nativeint.neg (Nativeint.of_string ("-" ^ String.sub s 0 (String.length s - 1)))

(* Remove underscores from float literals *)

let remove_underscores s =
  let l = String.length s in
  let rec remove src dst =
    if src >= l then
      if dst >= l then s else String.sub s 0 dst
    else
      match s.[src] with
        '_' -> remove (src + 1) dst
      |  c  -> s.[dst] <- c; remove (src + 1) (dst + 1)
  in remove 0 0

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

(* Error report *)

open Format

let report_error ppf = function
  | Illegal_character c ->
      fprintf ppf "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape s ->
      fprintf ppf "Illegal backslash escape in string or character (%s)" s
  | Unterminated_comment ->
      fprintf ppf "Comment not terminated"
  | Unterminated_string ->
      fprintf ppf "String literal not terminated"
  | Unterminated_string_in_comment ->
      fprintf ppf "This comment contains an unterminated string literal"
  | Keyword_as_label kwd ->
      fprintf ppf "`%s' is a keyword, it cannot be used as label name" kwd
  | Literal_overflow ty ->
      fprintf ppf "Integer literal exceeds the range of representable integers of type %s" ty
;;


# 206 "parsing/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\180\255\181\255\224\000\003\001\038\001\073\001\108\001\
    \196\255\143\001\180\001\032\000\204\255\067\000\217\001\252\001\
    \069\000\072\000\085\000\031\002\221\255\223\255\226\255\066\002\
    \123\000\101\002\093\000\047\001\240\255\120\002\153\002\226\002\
    \178\003\145\004\237\004\189\005\127\000\001\000\255\255\156\006\
    \186\006\251\255\138\007\105\008\248\255\241\255\242\255\243\255\
    \095\000\045\003\093\000\112\000\055\003\253\003\008\006\103\002\
    \173\004\133\000\122\008\098\000\237\000\114\000\239\255\238\255\
    \234\255\097\005\078\003\115\000\237\255\026\004\117\000\236\255\
    \022\006\118\000\235\255\117\000\232\255\147\008\231\255\044\007\
    \016\005\004\000\230\255\007\000\009\001\045\001\008\000\005\000\
    \230\255\214\008\249\008\030\009\065\009\218\255\214\255\215\255\
    \216\255\212\255\100\009\205\255\206\255\207\255\202\255\199\255\
    \135\009\195\255\197\255\170\009\205\009\240\009\103\003\244\255\
    \006\000\245\255\246\255\062\002\253\255\127\000\127\000\255\255\
    \254\255\002\010\131\000\255\000\176\000\252\255\251\255\250\255\
    \014\007\145\004\219\000\249\255\162\004\220\000\248\255\213\007\
    \229\000\247\255\077\004\247\255\248\255\009\000\249\255\075\010\
    \255\255\250\255\109\010\237\004\253\255\015\001\012\001\001\005\
    \252\255\148\010\251\255\237\000\240\000\018\000\027\001\246\000\
    \046\001\248\000\012\000\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\073\000\070\000\069\000\063\000\061\000\
    \255\255\057\000\054\000\052\000\255\255\047\000\046\000\044\000\
    \042\000\038\000\036\000\065\000\255\255\255\255\255\255\027\000\
    \026\000\033\000\031\000\030\000\255\255\010\000\010\000\009\000\
    \008\000\005\000\003\000\002\000\001\000\000\000\255\255\068\000\
    \255\255\255\255\255\255\006\000\255\255\255\255\255\255\255\255\
    \255\255\011\000\255\255\255\255\255\255\010\000\010\000\010\000\
    \011\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\021\000\021\000\021\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\022\000\255\255\072\000\255\255\255\255\
    \255\255\025\000\255\255\255\255\255\255\255\255\255\255\025\000\
    \255\255\028\000\071\000\066\000\035\000\255\255\255\255\255\255\
    \255\255\255\255\045\000\255\255\255\255\255\255\255\255\255\255\
    \055\000\255\255\255\255\067\000\062\000\064\000\255\255\255\255\
    \010\000\255\255\255\255\011\000\255\255\011\000\011\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\008\000\
    \255\255\255\255\005\000\005\000\255\255\001\000\001\000\255\255\
    \255\255\255\255\255\255\002\000\255\255\255\255\001\000\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\059\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\255\255\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\064\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\000\000\255\255\000\000\083\000\
    \255\255\255\255\000\000\083\000\084\000\083\000\086\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\255\255\255\255\255\255\111\000\000\000\
    \255\255\000\000\000\000\122\000\000\000\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \255\255\255\255\255\255\000\000\255\255\255\255\000\000\255\255\
    \255\255\000\000\139\000\000\000\000\000\255\255\000\000\145\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \000\000\255\255\000\000\255\255\255\255\157\000\160\000\255\255\
    \160\000\255\255\255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\036\000\038\000\038\000\036\000\037\000\082\000\088\000\
    \113\000\082\000\088\000\142\000\081\000\087\000\163\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\000\000\000\000\000\000\
    \036\000\007\000\028\000\024\000\005\000\003\000\023\000\027\000\
    \026\000\021\000\025\000\006\000\020\000\019\000\018\000\003\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\017\000\016\000\015\000\014\000\009\000\033\000\
    \004\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\013\000\102\000\012\000\004\000\035\000\
    \022\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\011\000\010\000\008\000\034\000\100\000\
    \097\000\099\000\096\000\093\000\080\000\095\000\094\000\075\000\
    \036\000\063\000\057\000\036\000\057\000\055\000\055\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\062\000\068\000\080\000\071\000\074\000\076\000\036\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \120\000\119\000\127\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\101\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\126\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\061\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \002\000\003\000\131\000\134\000\003\000\003\000\003\000\255\255\
    \255\255\124\000\003\000\003\000\137\000\003\000\003\000\003\000\
    \156\000\157\000\255\255\082\000\062\000\150\000\081\000\161\000\
    \150\000\150\000\003\000\162\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\004\000\159\000\126\000\004\000\
    \004\000\004\000\000\000\086\000\150\000\004\000\004\000\150\000\
    \004\000\004\000\004\000\000\000\000\000\000\000\085\000\082\000\
    \159\000\061\000\081\000\000\000\060\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\000\000\005\000\
    \000\000\000\000\005\000\005\000\005\000\085\000\000\000\084\000\
    \005\000\005\000\000\000\005\000\005\000\005\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\000\000\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\090\000\000\000\000\000\090\000\090\000\090\000\
    \000\000\000\000\000\000\090\000\090\000\000\000\090\000\109\000\
    \090\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \000\000\004\000\000\000\090\000\005\000\090\000\090\000\090\000\
    \090\000\090\000\000\000\058\000\000\000\107\000\000\000\000\000\
    \107\000\107\000\107\000\000\000\000\000\000\000\107\000\107\000\
    \000\000\107\000\107\000\107\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\005\000\000\000\005\000\000\000\107\000\090\000\
    \107\000\108\000\107\000\107\000\107\000\000\000\000\000\000\000\
    \005\000\000\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\005\000\005\000\000\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\090\000\000\000\090\000\
    \000\000\005\000\107\000\005\000\005\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\000\000\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \000\000\005\000\005\000\005\000\000\000\000\000\000\000\000\000\
    \107\000\000\000\107\000\000\000\106\000\005\000\005\000\000\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \000\000\000\000\005\000\000\000\000\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\005\000\005\000\000\000\005\000\005\000\
    \005\000\255\255\000\000\005\000\105\000\005\000\000\000\000\000\
    \000\000\103\000\005\000\005\000\000\000\005\000\005\000\005\000\
    \005\000\005\000\000\000\255\255\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \000\000\098\000\005\000\005\000\000\000\255\255\255\255\255\255\
    \104\000\000\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \090\000\000\000\000\000\090\000\090\000\090\000\000\000\000\000\
    \124\000\090\000\090\000\123\000\090\000\091\000\090\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\005\000\000\000\005\000\
    \000\000\090\000\005\000\090\000\090\000\092\000\090\000\090\000\
    \000\000\000\000\000\000\005\000\000\000\125\000\005\000\005\000\
    \089\000\000\000\000\000\000\000\005\000\005\000\000\000\005\000\
    \005\000\005\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \005\000\000\000\005\000\000\000\005\000\090\000\005\000\005\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\003\000\000\000\
    \000\000\003\000\003\000\003\000\000\000\000\000\078\000\077\000\
    \003\000\000\000\003\000\003\000\003\000\000\000\000\000\055\000\
    \055\000\000\000\121\000\090\000\000\000\090\000\000\000\003\000\
    \005\000\003\000\003\000\003\000\003\000\003\000\049\000\000\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\000\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\048\000\005\000\000\000\
    \005\000\000\000\000\000\003\000\046\000\000\000\055\000\049\000\
    \000\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\047\000\000\000\045\000\000\000\029\000\
    \000\000\000\000\000\000\050\000\000\000\048\000\048\000\000\000\
    \000\000\003\000\000\000\003\000\047\000\046\000\045\000\000\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\052\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \029\000\000\000\000\000\050\000\000\000\000\000\048\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\047\000\000\000\045\000\
    \051\000\031\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\052\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\000\000\255\255\000\000\
    \000\000\031\000\000\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\113\000\048\000\000\000\112\000\000\000\000\000\000\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \000\000\116\000\000\000\000\000\049\000\000\000\115\000\118\000\
    \000\000\117\000\048\000\000\000\000\000\000\000\000\000\000\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\000\000\000\000\
    \000\000\000\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\000\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\032\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\000\000\000\000\000\000\142\000\
    \000\000\000\000\141\000\000\000\053\000\000\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\000\000\000\000\000\000\114\000\
    \000\000\047\000\000\000\045\000\000\000\000\000\000\000\144\000\
    \000\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\143\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\039\000\000\000\000\000\039\000\039\000\039\000\
    \000\000\000\000\000\000\039\000\039\000\000\000\039\000\039\000\
    \039\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\039\000\000\000\039\000\039\000\039\000\
    \043\000\039\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\000\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\039\000\
    \042\000\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\056\000\039\000\039\000\039\000\
    \000\000\039\000\039\000\039\000\000\000\000\000\000\000\039\000\
    \039\000\080\000\039\000\039\000\039\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\039\000\
    \000\000\039\000\039\000\039\000\039\000\039\000\000\000\000\000\
    \080\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\000\000\000\000\000\000\000\000\000\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\000\000\039\000\040\000\140\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \000\000\039\000\000\000\039\000\000\000\000\000\000\000\000\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \000\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \000\000\000\000\000\000\000\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\032\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\000\000\000\000\000\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \000\000\000\000\000\000\000\000\046\000\000\000\000\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\054\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\047\000\000\000\045\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\039\000\000\000\000\000\
    \039\000\039\000\039\000\000\000\000\000\000\000\039\000\039\000\
    \000\000\039\000\039\000\039\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\039\000\000\000\
    \039\000\039\000\039\000\039\000\039\000\000\000\000\000\000\000\
    \000\000\040\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\041\000\000\000\000\000\000\000\
    \000\000\000\000\039\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\000\000\000\000\000\000\
    \039\000\040\000\039\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\085\000\082\000\000\000\
    \000\000\081\000\000\000\000\000\000\000\000\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \000\000\000\000\000\000\000\000\085\000\000\000\084\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\000\000\000\000\000\000\
    \000\000\000\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\000\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\042\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\000\000\000\000\000\000\
    \000\000\042\000\000\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\000\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\039\000\000\000\000\000\039\000\039\000\039\000\
    \000\000\000\000\000\000\039\000\039\000\000\000\039\000\039\000\
    \039\000\000\000\067\000\000\000\067\000\000\000\000\000\000\000\
    \000\000\067\000\000\000\039\000\000\000\039\000\039\000\039\000\
    \039\000\039\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\077\000\000\000\000\000\077\000\
    \077\000\077\000\000\000\000\000\000\000\077\000\077\000\000\000\
    \077\000\077\000\077\000\000\000\000\000\000\000\000\000\039\000\
    \000\000\000\000\000\000\000\000\000\000\077\000\000\000\077\000\
    \077\000\077\000\077\000\077\000\000\000\000\000\067\000\000\000\
    \000\000\000\000\000\000\000\000\067\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\039\000\000\000\039\000\
    \067\000\000\000\000\000\000\000\067\000\000\000\067\000\000\000\
    \000\000\077\000\065\000\000\000\000\000\000\000\000\000\005\000\
    \000\000\000\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \005\000\005\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \005\000\077\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\090\000\000\000\000\000\090\000\090\000\090\000\
    \000\000\000\000\000\000\090\000\090\000\000\000\090\000\090\000\
    \090\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\090\000\005\000\090\000\090\000\090\000\
    \090\000\090\000\000\000\000\000\000\000\000\000\000\000\090\000\
    \000\000\000\000\090\000\090\000\090\000\000\000\000\000\000\000\
    \090\000\090\000\000\000\090\000\090\000\090\000\000\000\000\000\
    \000\000\000\000\005\000\000\000\005\000\000\000\000\000\090\000\
    \090\000\000\000\090\000\090\000\090\000\090\000\090\000\000\000\
    \000\000\000\000\090\000\000\000\000\000\090\000\090\000\090\000\
    \000\000\000\000\000\000\090\000\090\000\000\000\090\000\090\000\
    \090\000\000\000\000\000\000\000\000\000\090\000\000\000\090\000\
    \000\000\000\000\255\255\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\000\000\000\000\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \000\000\005\000\005\000\005\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\090\000\000\000\090\000\000\000\005\000\090\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \005\000\000\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\005\000\005\000\000\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\090\000\000\000\090\000\
    \000\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\107\000\000\000\000\000\107\000\107\000\
    \107\000\000\000\000\000\000\000\107\000\107\000\000\000\107\000\
    \107\000\107\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \005\000\000\000\005\000\000\000\107\000\005\000\107\000\107\000\
    \107\000\107\000\107\000\000\000\000\000\000\000\107\000\000\000\
    \000\000\107\000\107\000\107\000\000\000\000\000\000\000\107\000\
    \107\000\000\000\107\000\107\000\107\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\005\000\000\000\005\000\000\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\000\000\000\000\
    \000\000\090\000\000\000\000\000\090\000\090\000\090\000\000\000\
    \000\000\000\000\090\000\090\000\000\000\090\000\090\000\090\000\
    \000\000\000\000\130\000\000\000\130\000\000\000\107\000\000\000\
    \107\000\130\000\090\000\107\000\090\000\090\000\090\000\090\000\
    \090\000\000\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\107\000\000\000\107\000\000\000\000\000\090\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\150\000\000\000\000\000\
    \149\000\000\000\000\000\000\000\000\000\000\000\130\000\000\000\
    \000\000\000\000\000\000\000\000\130\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\148\000\090\000\148\000\090\000\000\000\
    \130\000\000\000\148\000\000\000\130\000\000\000\130\000\000\000\
    \000\000\000\000\128\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\148\000\
    \000\000\000\000\000\000\000\000\000\000\148\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\000\000\000\000\000\000\000\000\
    \000\000\148\000\000\000\000\000\000\000\148\000\000\000\148\000\
    \000\000\000\000\000\000\146\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\000\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\037\000\000\000\000\000\081\000\087\000\
    \112\000\083\000\086\000\141\000\083\000\086\000\162\000\255\255\
    \255\255\255\255\255\255\255\255\157\000\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\013\000\
    \016\000\013\000\017\000\018\000\024\000\017\000\017\000\026\000\
    \036\000\059\000\048\000\036\000\048\000\050\000\050\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\061\000\067\000\024\000\070\000\073\000\075\000\036\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \117\000\118\000\122\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\013\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\124\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\060\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\130\000\133\000\003\000\003\000\003\000\083\000\
    \086\000\123\000\003\000\003\000\136\000\003\000\003\000\003\000\
    \155\000\156\000\157\000\084\000\060\000\150\000\084\000\159\000\
    \149\000\149\000\003\000\161\000\003\000\003\000\003\000\003\000\
    \003\000\255\255\255\255\255\255\004\000\158\000\123\000\004\000\
    \004\000\004\000\255\255\084\000\150\000\004\000\004\000\149\000\
    \004\000\004\000\004\000\255\255\255\255\255\255\085\000\085\000\
    \160\000\027\000\085\000\255\255\027\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\255\255\255\255\255\255\005\000\
    \255\255\255\255\005\000\005\000\005\000\085\000\255\255\085\000\
    \005\000\005\000\255\255\005\000\005\000\005\000\027\000\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\255\255\
    \255\255\255\255\006\000\255\255\255\255\006\000\006\000\006\000\
    \255\255\255\255\255\255\006\000\006\000\255\255\006\000\006\000\
    \006\000\255\255\255\255\255\255\255\255\255\255\255\255\004\000\
    \255\255\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\027\000\255\255\007\000\255\255\255\255\
    \007\000\007\000\007\000\255\255\255\255\255\255\007\000\007\000\
    \255\255\007\000\007\000\007\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \009\000\255\255\255\255\009\000\009\000\009\000\255\255\255\255\
    \255\255\009\000\009\000\255\255\009\000\009\000\009\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \255\255\009\000\007\000\009\000\009\000\009\000\009\000\009\000\
    \255\255\255\255\255\255\255\255\255\255\010\000\255\255\255\255\
    \010\000\010\000\010\000\255\255\255\255\255\255\010\000\010\000\
    \255\255\010\000\010\000\010\000\255\255\255\255\255\255\255\255\
    \007\000\255\255\007\000\255\255\009\000\009\000\010\000\255\255\
    \010\000\010\000\010\000\010\000\010\000\255\255\255\255\255\255\
    \255\255\255\255\014\000\255\255\255\255\014\000\014\000\014\000\
    \255\255\255\255\255\255\014\000\014\000\255\255\014\000\014\000\
    \014\000\084\000\255\255\009\000\009\000\009\000\255\255\255\255\
    \255\255\010\000\010\000\014\000\255\255\014\000\014\000\014\000\
    \014\000\014\000\255\255\158\000\255\255\015\000\255\255\255\255\
    \015\000\015\000\015\000\255\255\255\255\255\255\015\000\015\000\
    \255\255\015\000\015\000\015\000\255\255\085\000\160\000\027\000\
    \010\000\255\255\010\000\255\255\255\255\255\255\015\000\014\000\
    \015\000\015\000\015\000\015\000\015\000\255\255\255\255\255\255\
    \019\000\255\255\255\255\019\000\019\000\019\000\255\255\255\255\
    \115\000\019\000\019\000\115\000\019\000\019\000\019\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\014\000\255\255\014\000\
    \255\255\019\000\015\000\019\000\019\000\019\000\019\000\019\000\
    \255\255\255\255\255\255\023\000\255\255\115\000\023\000\023\000\
    \023\000\255\255\255\255\255\255\023\000\023\000\255\255\023\000\
    \023\000\023\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \015\000\255\255\015\000\255\255\023\000\019\000\023\000\023\000\
    \023\000\023\000\023\000\255\255\255\255\255\255\025\000\255\255\
    \255\255\025\000\025\000\025\000\255\255\255\255\025\000\025\000\
    \025\000\255\255\025\000\025\000\025\000\255\255\255\255\055\000\
    \055\000\255\255\115\000\019\000\255\255\019\000\255\255\025\000\
    \023\000\025\000\025\000\025\000\025\000\025\000\029\000\255\255\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\255\255\055\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\029\000\023\000\255\255\
    \023\000\255\255\255\255\025\000\029\000\255\255\055\000\030\000\
    \255\255\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\055\000\255\255\055\000\255\255\029\000\
    \255\255\255\255\255\255\030\000\255\255\029\000\030\000\255\255\
    \255\255\025\000\255\255\025\000\029\000\030\000\029\000\255\255\
    \030\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\030\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\000\255\255\255\255\030\000\255\255\255\255\030\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\030\000\255\255\030\000\
    \030\000\031\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\030\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\115\000\255\255\
    \255\255\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\110\000\049\000\255\255\110\000\255\255\255\255\255\255\
    \052\000\052\000\052\000\052\000\052\000\052\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\110\000\255\255\255\255\049\000\255\255\110\000\110\000\
    \255\255\110\000\049\000\255\255\255\255\255\255\255\255\255\255\
    \052\000\052\000\052\000\052\000\052\000\052\000\255\255\255\255\
    \255\255\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\032\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\053\000\053\000\
    \053\000\053\000\053\000\053\000\255\255\255\255\255\255\255\255\
    \255\255\053\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\255\255\255\255\255\255\138\000\
    \255\255\255\255\138\000\255\255\053\000\255\255\053\000\053\000\
    \053\000\053\000\053\000\053\000\255\255\255\255\255\255\110\000\
    \255\255\053\000\255\255\053\000\255\255\255\255\255\255\138\000\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\138\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\033\000\255\255\255\255\033\000\033\000\033\000\
    \255\255\255\255\255\255\033\000\033\000\255\255\033\000\033\000\
    \033\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\033\000\255\255\033\000\033\000\033\000\
    \033\000\033\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\255\255\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\056\000\033\000\034\000\033\000\
    \255\255\034\000\034\000\034\000\255\255\255\255\255\255\034\000\
    \034\000\080\000\034\000\034\000\034\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\147\000\147\000\034\000\
    \255\255\034\000\034\000\034\000\034\000\034\000\255\255\255\255\
    \080\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\255\255\255\255\255\255\255\255\255\255\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\255\255\034\000\034\000\138\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\034\000\255\255\034\000\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\065\000\065\000\065\000\065\000\065\000\065\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\065\000\065\000\065\000\065\000\065\000\065\000\
    \255\255\255\255\255\255\255\255\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\035\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\054\000\255\255\255\255\072\000\
    \072\000\072\000\072\000\072\000\072\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\054\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\054\000\255\255\054\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\039\000\255\255\255\255\
    \039\000\039\000\039\000\255\255\255\255\255\255\039\000\039\000\
    \255\255\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\039\000\255\255\
    \039\000\039\000\039\000\039\000\039\000\255\255\255\255\255\255\
    \255\255\040\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\255\255\255\255\255\255\
    \255\255\255\255\039\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\255\255\255\255\255\255\
    \039\000\040\000\039\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\079\000\079\000\255\255\
    \255\255\079\000\255\255\255\255\255\255\255\255\128\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\128\000\128\000\128\000\
    \255\255\255\255\255\255\255\255\079\000\255\255\079\000\128\000\
    \128\000\128\000\128\000\128\000\128\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\128\000\
    \128\000\128\000\128\000\128\000\128\000\255\255\255\255\255\255\
    \255\255\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\042\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\255\255\255\255\255\255\
    \255\255\042\000\255\255\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\135\000\135\000\
    \135\000\135\000\135\000\135\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\079\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\135\000\135\000\
    \135\000\135\000\135\000\135\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\043\000\255\255\255\255\043\000\043\000\043\000\
    \255\255\255\255\255\255\043\000\043\000\255\255\043\000\043\000\
    \043\000\255\255\058\000\255\255\058\000\255\255\255\255\255\255\
    \255\255\058\000\255\255\043\000\255\255\043\000\043\000\043\000\
    \043\000\043\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\077\000\255\255\255\255\077\000\
    \077\000\077\000\255\255\255\255\255\255\077\000\077\000\255\255\
    \077\000\077\000\077\000\255\255\255\255\255\255\255\255\043\000\
    \255\255\255\255\255\255\255\255\255\255\077\000\255\255\077\000\
    \077\000\077\000\077\000\077\000\255\255\255\255\058\000\255\255\
    \255\255\255\255\255\255\255\255\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\043\000\255\255\043\000\
    \058\000\255\255\255\255\255\255\058\000\255\255\058\000\255\255\
    \255\255\077\000\058\000\255\255\255\255\255\255\255\255\089\000\
    \255\255\255\255\089\000\089\000\089\000\255\255\255\255\255\255\
    \089\000\089\000\255\255\089\000\089\000\089\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\077\000\
    \089\000\077\000\089\000\089\000\089\000\089\000\089\000\255\255\
    \255\255\255\255\090\000\255\255\255\255\090\000\090\000\090\000\
    \255\255\255\255\255\255\090\000\090\000\255\255\090\000\090\000\
    \090\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\090\000\089\000\090\000\090\000\090\000\
    \090\000\090\000\255\255\255\255\255\255\255\255\255\255\091\000\
    \255\255\255\255\091\000\091\000\091\000\255\255\255\255\255\255\
    \091\000\091\000\255\255\091\000\091\000\091\000\255\255\255\255\
    \255\255\255\255\089\000\255\255\089\000\255\255\255\255\090\000\
    \091\000\255\255\091\000\091\000\091\000\091\000\091\000\255\255\
    \255\255\255\255\092\000\255\255\255\255\092\000\092\000\092\000\
    \255\255\255\255\255\255\092\000\092\000\255\255\092\000\092\000\
    \092\000\255\255\255\255\255\255\255\255\090\000\255\255\090\000\
    \255\255\255\255\058\000\092\000\091\000\092\000\092\000\092\000\
    \092\000\092\000\255\255\255\255\255\255\098\000\255\255\255\255\
    \098\000\098\000\098\000\255\255\255\255\255\255\098\000\098\000\
    \255\255\098\000\098\000\098\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\091\000\255\255\091\000\255\255\098\000\092\000\
    \098\000\098\000\098\000\098\000\098\000\255\255\255\255\255\255\
    \104\000\255\255\255\255\104\000\104\000\104\000\255\255\255\255\
    \255\255\104\000\104\000\255\255\104\000\104\000\104\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\092\000\255\255\092\000\
    \255\255\104\000\098\000\104\000\104\000\104\000\104\000\104\000\
    \255\255\255\255\255\255\107\000\255\255\255\255\107\000\107\000\
    \107\000\255\255\255\255\255\255\107\000\107\000\255\255\107\000\
    \107\000\107\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \098\000\255\255\098\000\255\255\107\000\104\000\107\000\107\000\
    \107\000\107\000\107\000\255\255\255\255\255\255\108\000\255\255\
    \255\255\108\000\108\000\108\000\255\255\255\255\255\255\108\000\
    \108\000\255\255\108\000\108\000\108\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\104\000\255\255\104\000\255\255\108\000\
    \107\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\109\000\255\255\255\255\109\000\109\000\109\000\255\255\
    \255\255\255\255\109\000\109\000\255\255\109\000\109\000\109\000\
    \255\255\255\255\121\000\255\255\121\000\255\255\107\000\255\255\
    \107\000\121\000\109\000\108\000\109\000\109\000\109\000\109\000\
    \109\000\255\255\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\121\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\108\000\255\255\108\000\255\255\255\255\109\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\143\000\255\255\255\255\
    \143\000\255\255\255\255\255\255\255\255\255\255\121\000\255\255\
    \255\255\255\255\255\255\255\255\121\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\143\000\109\000\143\000\109\000\255\255\
    \121\000\255\255\143\000\255\255\121\000\255\255\121\000\255\255\
    \255\255\255\255\121\000\143\000\143\000\143\000\143\000\143\000\
    \143\000\143\000\143\000\143\000\143\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\143\000\
    \255\255\255\255\255\255\255\255\255\255\143\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\255\255\255\255\255\255\255\255\
    \255\255\143\000\255\255\255\255\255\255\143\000\255\255\143\000\
    \255\255\255\255\255\255\143\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\255\255\153\000\153\000\153\000\
    \153\000\153\000\153\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\153\000\153\000\153\000\
    \153\000\153\000\153\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\143\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\010\000\
    \036\000\000\000\012\000\000\000\000\000\002\000\000\000\000\000\
    \027\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\002\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\000\000\000\000\000\000\000\000\000\000\027\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\039\000\039\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\019\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\024\000\084\000\143\000\149\000\084\000\143\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \024\000\255\255\084\000\000\000\085\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\079\000\080\000\255\255\255\255\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\080\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \084\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 8 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 246 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        token lexbuf
      )
# 1175 "parsing/lexer.ml"

  | 1 ->
# 250 "parsing/lexer.mll"
      ( token lexbuf )
# 1180 "parsing/lexer.ml"

  | 2 ->
# 252 "parsing/lexer.mll"
      ( UNDERSCORE )
# 1185 "parsing/lexer.ml"

  | 3 ->
# 254 "parsing/lexer.mll"
      ( TILDE )
# 1190 "parsing/lexer.ml"

  | 4 ->
# 256 "parsing/lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        let name = String.sub s 1 (String.length s - 2) in
        if Hashtbl.mem keyword_table name then
          raise (Error(Keyword_as_label name, Location.curr lexbuf));
        LABEL name )
# 1199 "parsing/lexer.ml"

  | 5 ->
# 261 "parsing/lexer.mll"
         ( QUESTION )
# 1204 "parsing/lexer.ml"

  | 6 ->
# 262 "parsing/lexer.mll"
         ( QUESTIONQUESTION )
# 1209 "parsing/lexer.ml"

  | 7 ->
# 264 "parsing/lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        let name = String.sub s 1 (String.length s - 2) in
        if Hashtbl.mem keyword_table name then
          raise (Error(Keyword_as_label name, Location.curr lexbuf));
        OPTLABEL name )
# 1218 "parsing/lexer.ml"

  | 8 ->
# 270 "parsing/lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
          try
            Hashtbl.find keyword_table s
          with Not_found ->
            LIDENT s )
# 1227 "parsing/lexer.ml"

  | 9 ->
# 276 "parsing/lexer.mll"
      ( UIDENT(Lexing.lexeme lexbuf) )
# 1232 "parsing/lexer.ml"

  | 10 ->
# 278 "parsing/lexer.mll"
      ( try
          INT (cvt_int_literal (Lexing.lexeme lexbuf))
        with Failure _ ->
          raise (Error(Literal_overflow "int", Location.curr lexbuf))
      )
# 1241 "parsing/lexer.ml"

  | 11 ->
# 284 "parsing/lexer.mll"
      ( FLOAT (remove_underscores(Lexing.lexeme lexbuf)) )
# 1246 "parsing/lexer.ml"

  | 12 ->
# 286 "parsing/lexer.mll"
      ( try
          INT32 (cvt_int32_literal (Lexing.lexeme lexbuf))
        with Failure _ ->
          raise (Error(Literal_overflow "int32", Location.curr lexbuf)) )
# 1254 "parsing/lexer.ml"

  | 13 ->
# 291 "parsing/lexer.mll"
      ( try
          INT64 (cvt_int64_literal (Lexing.lexeme lexbuf))
        with Failure _ ->
          raise (Error(Literal_overflow "int64", Location.curr lexbuf)) )
# 1262 "parsing/lexer.ml"

  | 14 ->
# 296 "parsing/lexer.mll"
      ( try
          NATIVEINT (cvt_nativeint_literal (Lexing.lexeme lexbuf))
        with Failure _ ->
          raise (Error(Literal_overflow "nativeint", Location.curr lexbuf)) )
# 1270 "parsing/lexer.ml"

  | 15 ->
# 301 "parsing/lexer.mll"
      ( reset_string_buffer();
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        string lexbuf;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string()) )
# 1280 "parsing/lexer.ml"

  | 16 ->
# 308 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        CHAR (Lexing.lexeme_char lexbuf 1) )
# 1286 "parsing/lexer.ml"

  | 17 ->
# 311 "parsing/lexer.mll"
      ( CHAR(Lexing.lexeme_char lexbuf 1) )
# 1291 "parsing/lexer.ml"

  | 18 ->
# 313 "parsing/lexer.mll"
      ( CHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) )
# 1296 "parsing/lexer.ml"

  | 19 ->
# 315 "parsing/lexer.mll"
      ( CHAR(char_for_decimal_code lexbuf 2) )
# 1301 "parsing/lexer.ml"

  | 20 ->
# 317 "parsing/lexer.mll"
      ( CHAR(char_for_hexadecimal_code lexbuf 3) )
# 1306 "parsing/lexer.ml"

  | 21 ->
# 319 "parsing/lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let esc = String.sub l 1 (String.length l - 1) in
        raise (Error(Illegal_escape esc, Location.curr lexbuf))
      )
# 1314 "parsing/lexer.ml"

  | 22 ->
# 324 "parsing/lexer.mll"
      ( comment_start_loc := [Location.curr lexbuf];
        comment lexbuf;
        token lexbuf )
# 1321 "parsing/lexer.ml"

  | 23 ->
# 328 "parsing/lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_start;
        comment_start_loc := [Location.curr lexbuf];
        comment lexbuf;
        token lexbuf
      )
# 1331 "parsing/lexer.ml"

  | 24 ->
# 335 "parsing/lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
        let curpos = lexbuf.lex_curr_p in
        lexbuf.lex_curr_p <- { curpos with pos_cnum = curpos.pos_cnum - 1 };
        STAR
      )
# 1342 "parsing/lexer.ml"

  | 25 ->
let
# 342 "parsing/lexer.mll"
                                   num
# 1348 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 343 "parsing/lexer.mll"
                                           name
# 1353 "parsing/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 345 "parsing/lexer.mll"
      ( update_loc lexbuf name (int_of_string num) true 0;
        token lexbuf
      )
# 1359 "parsing/lexer.ml"

  | 26 ->
# 348 "parsing/lexer.mll"
         ( SHARP )
# 1364 "parsing/lexer.ml"

  | 27 ->
# 349 "parsing/lexer.mll"
         ( AMPERSAND )
# 1369 "parsing/lexer.ml"

  | 28 ->
# 350 "parsing/lexer.mll"
         ( AMPERAMPER )
# 1374 "parsing/lexer.ml"

  | 29 ->
# 351 "parsing/lexer.mll"
         ( BACKQUOTE )
# 1379 "parsing/lexer.ml"

  | 30 ->
# 352 "parsing/lexer.mll"
         ( QUOTE )
# 1384 "parsing/lexer.ml"

  | 31 ->
# 353 "parsing/lexer.mll"
         ( LPAREN )
# 1389 "parsing/lexer.ml"

  | 32 ->
# 354 "parsing/lexer.mll"
         ( RPAREN )
# 1394 "parsing/lexer.ml"

  | 33 ->
# 355 "parsing/lexer.mll"
         ( STAR )
# 1399 "parsing/lexer.ml"

  | 34 ->
# 356 "parsing/lexer.mll"
         ( COMMA )
# 1404 "parsing/lexer.ml"

  | 35 ->
# 357 "parsing/lexer.mll"
         ( MINUSGREATER )
# 1409 "parsing/lexer.ml"

  | 36 ->
# 358 "parsing/lexer.mll"
         ( DOT )
# 1414 "parsing/lexer.ml"

  | 37 ->
# 359 "parsing/lexer.mll"
         ( DOTDOT )
# 1419 "parsing/lexer.ml"

  | 38 ->
# 360 "parsing/lexer.mll"
         ( COLON )
# 1424 "parsing/lexer.ml"

  | 39 ->
# 361 "parsing/lexer.mll"
         ( COLONCOLON )
# 1429 "parsing/lexer.ml"

  | 40 ->
# 362 "parsing/lexer.mll"
         ( COLONEQUAL )
# 1434 "parsing/lexer.ml"

  | 41 ->
# 363 "parsing/lexer.mll"
         ( COLONGREATER )
# 1439 "parsing/lexer.ml"

  | 42 ->
# 364 "parsing/lexer.mll"
         ( SEMI )
# 1444 "parsing/lexer.ml"

  | 43 ->
# 365 "parsing/lexer.mll"
         ( SEMISEMI )
# 1449 "parsing/lexer.ml"

  | 44 ->
# 366 "parsing/lexer.mll"
         ( LESS )
# 1454 "parsing/lexer.ml"

  | 45 ->
# 367 "parsing/lexer.mll"
         ( LESSMINUS )
# 1459 "parsing/lexer.ml"

  | 46 ->
# 368 "parsing/lexer.mll"
         ( EQUAL )
# 1464 "parsing/lexer.ml"

  | 47 ->
# 369 "parsing/lexer.mll"
         ( LBRACKET )
# 1469 "parsing/lexer.ml"

  | 48 ->
# 370 "parsing/lexer.mll"
         ( LBRACKETBAR )
# 1474 "parsing/lexer.ml"

  | 49 ->
# 371 "parsing/lexer.mll"
         ( LBRACKETLESS )
# 1479 "parsing/lexer.ml"

  | 50 ->
# 372 "parsing/lexer.mll"
         ( LBRACKETGREATER )
# 1484 "parsing/lexer.ml"

  | 51 ->
# 373 "parsing/lexer.mll"
         ( RBRACKET )
# 1489 "parsing/lexer.ml"

  | 52 ->
# 374 "parsing/lexer.mll"
         ( LBRACE )
# 1494 "parsing/lexer.ml"

  | 53 ->
# 375 "parsing/lexer.mll"
         ( LBRACELESS )
# 1499 "parsing/lexer.ml"

  | 54 ->
# 376 "parsing/lexer.mll"
         ( BAR )
# 1504 "parsing/lexer.ml"

  | 55 ->
# 377 "parsing/lexer.mll"
         ( BARBAR )
# 1509 "parsing/lexer.ml"

  | 56 ->
# 378 "parsing/lexer.mll"
         ( BARRBRACKET )
# 1514 "parsing/lexer.ml"

  | 57 ->
# 379 "parsing/lexer.mll"
         ( GREATER )
# 1519 "parsing/lexer.ml"

  | 58 ->
# 380 "parsing/lexer.mll"
         ( GREATERRBRACKET )
# 1524 "parsing/lexer.ml"

  | 59 ->
# 381 "parsing/lexer.mll"
         ( RBRACE )
# 1529 "parsing/lexer.ml"

  | 60 ->
# 382 "parsing/lexer.mll"
         ( GREATERRBRACE )
# 1534 "parsing/lexer.ml"

  | 61 ->
# 383 "parsing/lexer.mll"
         ( BANG )
# 1539 "parsing/lexer.ml"

  | 62 ->
# 385 "parsing/lexer.mll"
         ( INFIXOP0 "!=" )
# 1544 "parsing/lexer.ml"

  | 63 ->
# 386 "parsing/lexer.mll"
         ( PLUS )
# 1549 "parsing/lexer.ml"

  | 64 ->
# 387 "parsing/lexer.mll"
         ( PLUSDOT )
# 1554 "parsing/lexer.ml"

  | 65 ->
# 388 "parsing/lexer.mll"
         ( MINUS )
# 1559 "parsing/lexer.ml"

  | 66 ->
# 389 "parsing/lexer.mll"
         ( MINUSDOT )
# 1564 "parsing/lexer.ml"

  | 67 ->
# 392 "parsing/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 1569 "parsing/lexer.ml"

  | 68 ->
# 394 "parsing/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 1574 "parsing/lexer.ml"

  | 69 ->
# 396 "parsing/lexer.mll"
            ( INFIXOP0(Lexing.lexeme lexbuf) )
# 1579 "parsing/lexer.ml"

  | 70 ->
# 398 "parsing/lexer.mll"
            ( INFIXOP1(Lexing.lexeme lexbuf) )
# 1584 "parsing/lexer.ml"

  | 71 ->
# 400 "parsing/lexer.mll"
            ( INFIXOP2(Lexing.lexeme lexbuf) )
# 1589 "parsing/lexer.ml"

  | 72 ->
# 402 "parsing/lexer.mll"
            ( INFIXOP4(Lexing.lexeme lexbuf) )
# 1594 "parsing/lexer.ml"

  | 73 ->
# 404 "parsing/lexer.mll"
            ( INFIXOP3(Lexing.lexeme lexbuf) )
# 1599 "parsing/lexer.ml"

  | 74 ->
# 405 "parsing/lexer.mll"
        ( EOF )
# 1604 "parsing/lexer.ml"

  | 75 ->
# 407 "parsing/lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      )
# 1611 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 110
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 413 "parsing/lexer.mll"
      ( comment_start_loc := (Location.curr lexbuf) :: !comment_start_loc;
        comment lexbuf;
      )
# 1624 "parsing/lexer.ml"

  | 1 ->
# 417 "parsing/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | [x] -> comment_start_loc := [];
        | _ :: l -> comment_start_loc := l;
                    comment lexbuf;
       )
# 1634 "parsing/lexer.ml"

  | 2 ->
# 424 "parsing/lexer.mll"
      ( reset_string_buffer();
        string_start_loc := Location.curr lexbuf;
        begin try string lexbuf
        with Error (Unterminated_string, _) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ -> comment_start_loc := [];
                        raise (Error (Unterminated_string_in_comment, loc))
        end;
        reset_string_buffer ();
        comment lexbuf )
# 1649 "parsing/lexer.ml"

  | 3 ->
# 436 "parsing/lexer.mll"
      ( comment lexbuf )
# 1654 "parsing/lexer.ml"

  | 4 ->
# 438 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        comment lexbuf
      )
# 1661 "parsing/lexer.ml"

  | 5 ->
# 442 "parsing/lexer.mll"
      ( comment lexbuf )
# 1666 "parsing/lexer.ml"

  | 6 ->
# 444 "parsing/lexer.mll"
      ( comment lexbuf )
# 1671 "parsing/lexer.ml"

  | 7 ->
# 446 "parsing/lexer.mll"
      ( comment lexbuf )
# 1676 "parsing/lexer.ml"

  | 8 ->
# 448 "parsing/lexer.mll"
      ( comment lexbuf )
# 1681 "parsing/lexer.ml"

  | 9 ->
# 450 "parsing/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | loc :: _ -> comment_start_loc := [];
                      raise (Error (Unterminated_comment, loc))
      )
# 1690 "parsing/lexer.ml"

  | 10 ->
# 456 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        comment lexbuf
      )
# 1697 "parsing/lexer.ml"

  | 11 ->
# 460 "parsing/lexer.mll"
      ( comment lexbuf )
# 1702 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 2 (-1) ;   __ocaml_lex_string_rec lexbuf 138
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 464 "parsing/lexer.mll"
      ( () )
# 1713 "parsing/lexer.ml"

  | 1 ->
let
# 465 "parsing/lexer.mll"
                                  space
# 1719 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 466 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        string lexbuf
      )
# 1725 "parsing/lexer.ml"

  | 2 ->
# 470 "parsing/lexer.mll"
      ( store_string_char(char_for_backslash(Lexing.lexeme_char lexbuf 1));
        string lexbuf )
# 1731 "parsing/lexer.ml"

  | 3 ->
# 473 "parsing/lexer.mll"
      ( store_string_char(char_for_decimal_code lexbuf 1);
         string lexbuf )
# 1737 "parsing/lexer.ml"

  | 4 ->
# 476 "parsing/lexer.mll"
      ( store_string_char(char_for_hexadecimal_code lexbuf 2);
         string lexbuf )
# 1743 "parsing/lexer.ml"

  | 5 ->
# 479 "parsing/lexer.mll"
      ( if in_comment ()
        then string lexbuf
        else begin
(*  Should be an error, but we are very lax.
          raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
                        Location.curr lexbuf))
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
          store_string_char (Lexing.lexeme_char lexbuf 0);
          store_string_char (Lexing.lexeme_char lexbuf 1);
          string lexbuf
        end
      )
# 1761 "parsing/lexer.ml"

  | 6 ->
# 494 "parsing/lexer.mll"
      ( if not (in_comment ()) then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string;
        update_loc lexbuf None 1 false 0;
        let s = Lexing.lexeme lexbuf in
        for i = 0 to String.length s - 1 do
          store_string_char s.[i];
        done;
        string lexbuf
      )
# 1774 "parsing/lexer.ml"

  | 7 ->
# 504 "parsing/lexer.mll"
      ( raise (Error (Unterminated_string, !string_start_loc)) )
# 1779 "parsing/lexer.ml"

  | 8 ->
# 506 "parsing/lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        string lexbuf )
# 1785 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and skip_sharp_bang lexbuf =
    __ocaml_lex_skip_sharp_bang_rec lexbuf 155
and __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 511 "parsing/lexer.mll"
       ( update_loc lexbuf None 3 false 0 )
# 1796 "parsing/lexer.ml"

  | 1 ->
# 513 "parsing/lexer.mll"
       ( update_loc lexbuf None 1 false 0 )
# 1801 "parsing/lexer.ml"

  | 2 ->
# 514 "parsing/lexer.mll"
       ( () )
# 1806 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state

;;

